<?php
/* =========================================================
   ONE FILE Dynamic Admin Panel + API (JSON Storage)
   - Create Categories
   - Create Fields under Category (input builder)
   - Add/Edit/Delete Items under Category
   - Image upload (saved in uploads/ + stored in JSON)
   - Android/API: ?api=1 (returns full schema + data)
   ========================================================= */

date_default_timezone_set("Asia/Dhaka");

// ---------- CONFIG ----------
$DB_FILE = __DIR__ . "/db.json";
$UPLOAD_DIR = __DIR__ . "/uploads/";
$BASE_URL = (isset($_SERVER['HTTPS']) ? "https" : "http") . "://".$_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['SCRIPT_NAME']), "/\\") . "/";

// ---------- CORS for API ----------
if (isset($_GET["api"])) {
  header("Content-Type: application/json; charset=utf-8");
  header("Access-Control-Allow-Origin: *");
  header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
  header("Access-Control-Allow-Headers: Content-Type");
  if ($_SERVER["REQUEST_METHOD"] === "OPTIONS") { http_response_code(204); exit; }
}

// ---------- Helpers ----------
function jread($file){
  if (!file_exists($file)) return [];
  $raw = file_get_contents($file);
  $data = json_decode($raw, true);
  return is_array($data) ? $data : [];
}
function jwrite($file, $data){
  file_put_contents($file, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT));
}
function new_id($len=10){ return substr(bin2hex(random_bytes(16)), 0, $len); }
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, "UTF-8"); }
function slugify($t){
  $t = strtolower(trim($t));
  $t = preg_replace('/[^a-z0-9]+/','_',$t);
  $t = trim($t,'_');
  return $t ?: "cat_" . new_id(6);
}

// ---------- INIT DB ----------
if (!file_exists($DB_FILE)) {
  $init = [
    "categories" => [],
    "fields" => [],
    "items" => [],
    "meta" => [
      "breaking_news" => "Welcome!",
      "slider_speed" => 3000
    ]
  ];
  jwrite($DB_FILE, $init);
}

// ---------- Load DB ----------
$db = jread($DB_FILE);
if (!isset($db["categories"])) $db["categories"] = [];
if (!isset($db["fields"])) $db["fields"] = [];
if (!isset($db["items"])) $db["items"] = [];
if (!isset($db["meta"])) $db["meta"] = ["breaking_news"=>"Welcome!","slider_speed"=>3000];

if (!is_dir($UPLOAD_DIR)) @mkdir($UPLOAD_DIR, 0777, true);

// ---------- API MODE ----------
if (isset($_GET["api"])) {
  echo json_encode([
    "ok" => true,
    "base_url" => $BASE_URL,
    "meta" => $db["meta"],
    "categories" => $db["categories"],
    "fields" => $db["fields"],
    "items" => $db["items"],
  ], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

// ---------- ACTION HANDLER ----------
$action = $_POST["action"] ?? $_GET["action"] ?? "";
$msg = "";

// Create Category
if ($action === "create_category") {
  $name = trim($_POST["name"] ?? "");
  if ($name !== "") {
    $catId = new_id(8);
    $slug = slugify($_POST["slug"] ?? $name);
    $icon = trim($_POST["icon"] ?? "");
    $db["categories"][] = [
      "id" => $catId,
      "name" => $name,
      "slug" => $slug,
      "icon" => $icon,
      "created_at" => date("c")
    ];
    if (!isset($db["fields"][$catId])) $db["fields"][$catId] = [];
    if (!isset($db["items"][$catId])) $db["items"][$catId] = [];
    jwrite($DB_FILE, $db);
    $msg = "✅ Category added";
  } else $msg = "❌ Category name required";
}

// Delete Category
if ($action === "delete_category") {
  $catId = $_GET["id"] ?? "";
  if ($catId) {
    $db["categories"] = array_values(array_filter($db["categories"], fn($c)=>($c["id"]??"")!==$catId));
    unset($db["fields"][$catId]);
    unset($db["items"][$catId]);
    jwrite($DB_FILE, $db);
    $msg = "✅ Category deleted";
  }
}

// Add Field
if ($action === "add_field") {
  $catId = $_POST["cat_id"] ?? "";
  $key = trim($_POST["key"] ?? "");
  $label = trim($_POST["label"] ?? "");
  $type = $_POST["type"] ?? "text";
  $required = isset($_POST["required"]) ? true : false;
  $options = trim($_POST["options"] ?? "");

  if ($catId && $key && $label) {
    if (!isset($db["fields"][$catId])) $db["fields"][$catId] = [];
    $field = [
      "key" => $key,
      "label" => $label,
      "type" => $type,
      "required" => $required
    ];
    if ($type === "select") {
      $opts = array_values(array_filter(array_map("trim", explode(",", $options))));
      $field["options"] = $opts;
    }
    $db["fields"][$catId][] = $field;
    jwrite($DB_FILE, $db);
    $msg = "✅ Field added";
  } else $msg = "❌ Field info missing";
}

// Delete Field
if ($action === "delete_field") {
  $catId = $_GET["cat"] ?? "";
  $key = $_GET["key"] ?? "";
  if ($catId && $key && isset($db["fields"][$catId])) {
    $db["fields"][$catId] = array_values(array_filter($db["fields"][$catId], fn($f)=>($f["key"]??"")!==$key));
    jwrite($DB_FILE, $db);
    $msg = "✅ Field deleted";
  }
}

// Save Meta (Breaking news, slider speed)
if ($action === "save_meta") {
  $bn = trim($_POST["breaking_news"] ?? "");
  $speed = (int)($_POST["slider_speed"] ?? 3000);
  if ($speed < 1000) $speed = 1000;
  if ($speed > 20000) $speed = 20000;
  $db["meta"]["breaking_news"] = $bn;
  $db["meta"]["slider_speed"] = $speed;
  jwrite($DB_FILE, $db);
  $msg = "✅ Meta saved";
}

// Add Item
if ($action === "add_item") {
  $catId = $_POST["cat_id"] ?? "";
  if ($catId && isset($db["fields"][$catId])) {
    $item = ["id"=>new_id(10), "status"=>($_POST["status"]??"published"), "created_at"=>date("c")];

    foreach ($db["fields"][$catId] as $f) {
      $k = $f["key"];
      $t = $f["type"] ?? "text";

      if ($t === "image") {
        if (!empty($_FILES[$k]["name"])) {
          $ext = strtolower(pathinfo($_FILES[$k]["name"], PATHINFO_EXTENSION));
          $allow = ["jpg","jpeg","png","webp"];
          if (in_array($ext,$allow,true)) {
            $fname = $catId . "_" . $k . "_" . date("Ymd_His") . "_" . new_id(6) . "." . $ext;
            $path = $UPLOAD_DIR . $fname;
            if (move_uploaded_file($_FILES[$k]["tmp_name"], $path)) {
              $item[$k] = "uploads/" . $fname;
            }
          }
        }
      } else {
        $item[$k] = trim($_POST[$k] ?? "");
      }
    }

    if (!isset($db["items"][$catId])) $db["items"][$catId] = [];
    $db["items"][$catId][] = $item;
    jwrite($DB_FILE, $db);
    $msg = "✅ Item added";
  } else $msg = "❌ Invalid category";
}

// Delete Item
if ($action === "delete_item") {
  $catId = $_GET["cat"] ?? "";
  $id = $_GET["id"] ?? "";
  if ($catId && $id && isset($db["items"][$catId])) {
    $db["items"][$catId] = array_values(array_filter($db["items"][$catId], fn($it)=>($it["id"]??"")!==$id));
    jwrite($DB_FILE, $db);
    $msg = "✅ Item deleted";
  }
}

// Toggle Item Status
if ($action === "toggle_item") {
  $catId = $_GET["cat"] ?? "";
  $id = $_GET["id"] ?? "";
  if ($catId && $id && isset($db["items"][$catId])) {
    foreach ($db["items"][$catId] as &$it) {
      if (($it["id"] ?? "") === $id) {
        $it["status"] = (($it["status"] ?? "published") === "published") ? "unpublished" : "published";
        $it["updated_at"] = date("c");
        break;
      }
    }
    jwrite($DB_FILE, $db);
    $msg = "✅ Status updated";
  }
}

// ---------- UI ----------
$selectedCat = $_GET["cat"] ?? ($db["categories"][0]["id"] ?? "");
$fields = ($selectedCat && isset($db["fields"][$selectedCat])) ? $db["fields"][$selectedCat] : [];
$items = ($selectedCat && isset($db["items"][$selectedCat])) ? $db["items"][$selectedCat] : [];

?><!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Dynamic Admin Panel (JSON)</title>
<style>
  body{font-family:system-ui,Arial;margin:0;background:#f5f7fb;color:#111827}
  header{background:#111827;color:#fff;padding:14px 18px}
  .wrap{display:flex;gap:12px;padding:12px}
  .side{width:280px;background:#fff;border-radius:14px;padding:12px;box-shadow:0 10px 30px rgba(0,0,0,.06);height:calc(100vh - 90px);overflow:auto}
  .main{flex:1;background:#fff;border-radius:14px;padding:12px;box-shadow:0 10px 30px rgba(0,0,0,.06);height:calc(100vh - 90px);overflow:auto}
  h2{margin:10px 0 6px}
  .card{background:#f9fafb;border:1px solid #e5e7eb;border-radius:12px;padding:10px;margin:10px 0}
  input,select,textarea{width:100%;padding:10px;border-radius:10px;border:1px solid #d1d5db;margin:6px 0}
  button{padding:10px 12px;border:0;border-radius:10px;background:#2563eb;color:#fff;cursor:pointer}
  a.btn{display:inline-block;padding:8px 10px;border-radius:10px;background:#111827;color:#fff;text-decoration:none}
  .small{font-size:12px;color:#6b7280}
  .msg{padding:10px;border-radius:12px;margin:10px 0;background:#ecfeff;border:1px solid #a5f3fc}
  .cat{padding:10px;border-radius:12px;border:1px solid #e5e7eb;margin:8px 0}
  .cat.active{border-color:#2563eb;background:#eff6ff}
  table{width:100%;border-collapse:collapse}
  th,td{padding:10px;border-bottom:1px solid #e5e7eb;text-align:left;font-size:14px}
  .tag{padding:4px 8px;border-radius:999px;font-size:12px}
  .pub{background:#dcfce7;color:#166534}
  .unpub{background:#fee2e2;color:#991b1b}
</style>
</head>
<body>
<header>
  <div style="display:flex;justify-content:space-between;align-items:center;gap:10px;">
    <div><b>Dynamic Admin Panel</b> <span class="small">(JSON storage, single file)</span></div>
    <div class="small">API: <code><?php echo h($BASE_URL); ?>panel.php?api=1</code></div>
  </div>
</header>

<div class="wrap">
  <div class="side">
    <?php if($msg): ?><div class="msg"><?php echo h($msg); ?></div><?php endif; ?>

    <h2>Categories</h2>
    <?php foreach($db["categories"] as $c): 
      $active = (($c["id"]??"") === $selectedCat) ? "active" : "";
    ?>
      <div class="cat <?php echo $active; ?>">
        <div style="display:flex;justify-content:space-between;gap:10px;">
          <a href="?cat=<?php echo h($c["id"]); ?>" style="text-decoration:none;color:#111827;font-weight:700">
            <?php echo h($c["name"]); ?>
          </a>
          <a class="btn" style="background:#ef4444" href="?action=delete_category&id=<?php echo h($c["id"]); ?>" onclick="return confirm('Delete category?')">Del</a>
        </div>
        <div class="small">Slug: <?php echo h($c["slug"]); ?></div>
      </div>
    <?php endforeach; ?>

    <div class="card">
      <b>Add Category</b>
      <form method="post">
        <input type="hidden" name="action" value="create_category">
        <input name="name" placeholder="Category name (Ambulance)">
        <input name="slug" placeholder="slug (ambulance)">
        <input name="icon" placeholder="icon (optional)">
        <button type="submit">Create</button>
      </form>
    </div>

    <div class="card">
      <b>App Meta</b>
      <form method="post">
        <input type="hidden" name="action" value="save_meta">
        <textarea name="breaking_news" placeholder="Breaking news"><?php echo h($db["meta"]["breaking_news"] ?? ""); ?></textarea>
        <input name="slider_speed" type="number" min="1000" max="20000" value="<?php echo h($db["meta"]["slider_speed"] ?? 3000); ?>">
        <button type="submit">Save</button>
      </form>
    </div>
  </div>

  <div class="main">
    <h2>Selected Category</h2>
    <?php if(!$selectedCat): ?>
      <div class="card">No category. Create one first.</div>
    <?php else: ?>

      <h2>Fields Builder</h2>
      <div class="card">
        <form method="post">
          <input type="hidden" name="action" value="add_field">
          <input type="hidden" name="cat_id" value="<?php echo h($selectedCat); ?>">
          <input name="key" placeholder="key (name / phone / area)">
          <input name="label" placeholder="label (নাম / ফোন / এলাকা)">
          <select name="type">
            <option value="text">text</option>
            <option value="phone">phone</option>
            <option value="url">url</option>
            <option value="textarea">textarea</option>
            <option value="select">select</option>
            <option value="image">image</option>
          </select>
          <input name="options" placeholder="select options (a,b,c) only for select">
          <label class="small"><input type="checkbox" name="required"> required</label><br><br>
          <button type="submit">Add Field</button>
        </form>
      </div>

      <h2>Add Item</h2>
      <div class="card">
        <form method="post" enctype="multipart/form-data">
          <input type="hidden" name="action" value="add_item">
          <input type="hidden" name="cat_id" value="<?php echo h($selectedCat); ?>">

          <?php foreach($fields as $f):
            $k = $f["key"]; $t = $f["type"] ?? "text"; $lbl = $f["label"] ?? $k;
          ?>
            <label><b><?php echo h($lbl); ?></b></label>
            <?php if($t==="textarea"): ?>
              <textarea name="<?php echo h($k); ?>" placeholder="<?php echo h($lbl); ?>"></textarea>
            <?php elseif($t==="select"): ?>
              <select name="<?php echo h($k); ?>">
                <?php foreach(($f["options"] ?? []) as $opt): ?>
                  <option value="<?php echo h($opt); ?>"><?php echo h($opt); ?></option>
                <?php endforeach; ?>
              </select>
            <?php elseif($t==="image"): ?>
              <input type="file" name="<?php echo h($k); ?>" accept="image/*">
            <?php else: ?>
              <input name="<?php echo h($k); ?>" placeholder="<?php echo h($lbl); ?>">
            <?php endif; ?>
          <?php endforeach; ?>

          <label><b>Status</b></label>
          <select name="status">
            <option value="published">published</option>
            <option value="unpublished">unpublished</option>
          </select>

          <button type="submit">Save Item</button>
        </form>
      </div>

      <h2>Items</h2>
      <?php if(count($items)===0): ?>
        <div class="card">No items yet.</div>
      <?php else: ?>
        <table>
          <thead>
          <tr>
            <th>ID</th>
            <th>Status</th>
            <th>Data</th>
            <th>Action</th>
          </tr>
          </thead>
          <tbody>
          <?php foreach(array_reverse($items) as $it): ?>
            <tr>
              <td><?php echo h($it["id"] ?? ""); ?></td>
              <td><?php $st = $it["status"] ?? "published"; ?>
                <span class="tag <?php echo ($st==="published")?"pub":"unpub"; ?>"><?php echo h($st); ?></span>
              </td>
              <td class="small">
                <?php foreach($it as $kk=>$vv):
                  if(in_array($kk, ["id","status","created_at","updated_at"], true)) continue;
                  if(is_string($vv) && str_starts_with($vv,"uploads/")) {
                    echo "<div><b>".h($kk).":</b> <a href='".h($BASE_URL.$vv)."' target='_blank'>view</a></div>";
                  } else {
                    echo "<div><b>".h($kk).":</b> ".h($vv)."</div>";
                  }
                endforeach; ?>
              </td>
              <td>
                <a class="btn" href="?cat=<?php echo h($selectedCat); ?>&action=toggle_item&id=<?php echo h($it["id"]); ?>&cat=<?php echo h($selectedCat); ?>">Toggle</a>
                <a class="btn" style="background:#ef4444" href="?cat=<?php echo h($selectedCat); ?>&action=delete_item&id=<?php echo h($it["id"]); ?>&cat=<?php echo h($selectedCat); ?>" onclick="return confirm('Delete item?')">Del</a>
              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>
      <?php endif; ?>

    <?php endif; ?>
  </div>
</div>
</body>
</html>
