import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { toast } from "sonner";
import {
  Plus,
  Trash2,
  GripVertical,
  Image as ImageIcon,
  Link,
  Eye,
  EyeOff,
  Upload,
} from "lucide-react";

interface SliderItem {
  id: string;
  imageUrl: string;
  title: string;
  link: string;
  order: number;
  isActive: boolean;
}

const mockSliders: SliderItem[] = [
  {
    id: "1",
    imageUrl: "https://picsum.photos/seed/slider1/1200/400",
    title: "স্বাগতম শীতলকার কমিউনিটি",
    link: "",
    order: 1,
    isActive: true,
  },
  {
    id: "2",
    imageUrl: "https://picsum.photos/seed/slider2/1200/400",
    title: "রক্তদান করুন, জীবন বাঁচান",
    link: "/blood-donors",
    order: 2,
    isActive: true,
  },
  {
    id: "3",
    imageUrl: "https://picsum.photos/seed/slider3/1200/400",
    title: "জরুরি প্রয়োজনে কল করুন",
    link: "/emergency",
    order: 3,
    isActive: false,
  },
];

export default function SliderPage() {
  const [sliders, setSliders] = useState<SliderItem[]>(mockSliders);

  const handleAddSlider = () => {
    const newSlider: SliderItem = {
      id: String(Date.now()),
      imageUrl: "",
      title: "",
      link: "",
      order: sliders.length + 1,
      isActive: true,
    };
    setSliders([...sliders, newSlider]);
  };

  const handleDeleteSlider = (id: string) => {
    setSliders(sliders.filter((s) => s.id !== id));
    toast.success("স্লাইডার সফলভাবে মুছে ফেলা হয়েছে");
  };

  const handleUpdateSlider = (id: string, updates: Partial<SliderItem>) => {
    setSliders(
      sliders.map((s) => (s.id === id ? { ...s, ...updates } : s))
    );
  };

  const handleSave = () => {
    toast.success("স্লাইডার সফলভাবে সংরক্ষণ করা হয়েছে");
  };

  return (
    <AdminLayout
      title="স্লাইডার ম্যানেজমেন্ট"
      description="হোম পেজের স্লাইডার ইমেজ পরিচালনা করুন"
    >
      <div className="space-y-6">
        {/* Header Actions */}
        <div className="flex justify-between items-center">
          <p className="text-muted-foreground">
            মোট {sliders.length} টি স্লাইডার | সক্রিয় {sliders.filter((s) => s.isActive).length} টি
          </p>
          <div className="flex gap-2">
            <Button onClick={handleAddSlider} className="btn-gradient-primary gap-2">
              <Plus className="h-4 w-4" />
              নতুন স্লাইডার যুক্ত করুন
            </Button>
          </div>
        </div>

        {/* Slider List */}
        <div className="space-y-4">
          {sliders.map((slider, index) => (
            <Card key={slider.id} className="glass-card overflow-hidden">
              <div className="flex">
                {/* Drag Handle */}
                <div className="flex items-center justify-center w-12 bg-muted/50 cursor-move">
                  <GripVertical className="h-5 w-5 text-muted-foreground" />
                </div>

                {/* Image Preview */}
                <div className="w-48 h-32 bg-muted flex-shrink-0 relative group">
                  {slider.imageUrl ? (
                    <img
                      src={slider.imageUrl}
                      alt={slider.title}
                      className="w-full h-full object-cover"
                    />
                  ) : (
                    <div className="w-full h-full flex items-center justify-center">
                      <ImageIcon className="h-10 w-10 text-muted-foreground/50" />
                    </div>
                  )}
                  <div className="absolute inset-0 bg-foreground/50 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                    <Button size="sm" variant="secondary" className="gap-2">
                      <Upload className="h-4 w-4" />
                      আপলোড
                    </Button>
                  </div>
                </div>

                {/* Content */}
                <CardContent className="flex-1 p-4 grid gap-4 md:grid-cols-3">
                  <div className="space-y-2">
                    <Label className="text-xs text-muted-foreground">শিরোনাম</Label>
                    <Input
                      value={slider.title}
                      onChange={(e) =>
                        handleUpdateSlider(slider.id, { title: e.target.value })
                      }
                      placeholder="স্লাইডার শিরোনাম"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-xs text-muted-foreground">ইমেজ URL</Label>
                    <Input
                      value={slider.imageUrl}
                      onChange={(e) =>
                        handleUpdateSlider(slider.id, { imageUrl: e.target.value })
                      }
                      placeholder="https://..."
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-xs text-muted-foreground">লিংক (ঐচ্ছিক)</Label>
                    <div className="relative">
                      <Link className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                      <Input
                        value={slider.link}
                        onChange={(e) =>
                          handleUpdateSlider(slider.id, { link: e.target.value })
                        }
                        placeholder="/page-link"
                        className="pl-9"
                      />
                    </div>
                  </div>
                </CardContent>

                {/* Actions */}
                <div className="flex items-center gap-4 p-4 border-l border-border">
                  <div className="flex items-center gap-2">
                    {slider.isActive ? (
                      <Eye className="h-4 w-4 text-success" />
                    ) : (
                      <EyeOff className="h-4 w-4 text-muted-foreground" />
                    )}
                    <Switch
                      checked={slider.isActive}
                      onCheckedChange={(checked) =>
                        handleUpdateSlider(slider.id, { isActive: checked })
                      }
                    />
                  </div>

                  <Button
                    variant="ghost"
                    size="icon"
                    className="text-destructive hover:text-destructive hover:bg-destructive/10"
                    onClick={() => handleDeleteSlider(slider.id)}
                  >
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </Card>
          ))}
        </div>

        {/* Empty State */}
        {sliders.length === 0 && (
          <Card className="glass-card">
            <CardContent className="py-12 text-center">
              <ImageIcon className="h-12 w-12 mx-auto text-muted-foreground/50 mb-4" />
              <h3 className="font-medium mb-2">কোনো স্লাইডার নেই</h3>
              <p className="text-sm text-muted-foreground mb-4">
                নতুন স্লাইডার যুক্ত করতে উপরের বাটনে ক্লিক করুন
              </p>
              <Button onClick={handleAddSlider} className="btn-gradient-primary gap-2">
                <Plus className="h-4 w-4" />
                নতুন স্লাইডার যুক্ত করুন
              </Button>
            </CardContent>
          </Card>
        )}

        {/* Save Button */}
        {sliders.length > 0 && (
          <div className="flex justify-end">
            <Button onClick={handleSave} className="btn-gradient-primary">
              সংরক্ষণ করুন
            </Button>
          </div>
        )}
      </div>
    </AdminLayout>
  );
}
