import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import { Police } from "@/types";
import { Phone, MapPin, Shield, Building2 } from "lucide-react";

// Mock data
const mockPolice: Police[] = [
  {
    id: "1",
    name: "এস.আই. মোঃ রফিক",
    designation: "সাব-ইন্সপেক্টর",
    station: "শীতলকার থানা",
    area: "শীতলকার, বরিশাল",
    phone: "০১৭১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "2",
    name: "ও.সি. আব্দুল করিম",
    designation: "অফিসার ইনচার্জ",
    station: "বানারীপাড়া থানা",
    area: "বানারীপাড়া",
    phone: "০১৮১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "3",
    name: "কনস্টেবল জাহিদ",
    designation: "কনস্টেবল",
    station: "উজিরপুর থানা",
    area: "উজিরপুর",
    phone: "০১৯১২৩৪৫৬৭৮",
    status: "pending",
  },
];

const columns: Column<Police>[] = [
  {
    key: "name",
    header: "নাম",
    render: (item) => (
      <div className="flex items-center gap-3">
        <div className="h-10 w-10 rounded-full bg-info/10 flex items-center justify-center">
          <Shield className="h-5 w-5 text-info" />
        </div>
        <div>
          <p className="font-medium">{item.name}</p>
          <p className="text-sm text-muted-foreground">{item.designation}</p>
        </div>
      </div>
    ),
  },
  {
    key: "station",
    header: "থানা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Building2 className="h-4 w-4 text-muted-foreground" />
        <span>{item.station}</span>
      </div>
    ),
  },
  {
    key: "area",
    header: "এলাকা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.area}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function PolicePage() {
  const [police, setPolice] = useState<Police[]>(mockPolice);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingPolice, setEditingPolice] = useState<Police | null>(null);
  const [formData, setFormData] = useState({
    name: "",
    designation: "",
    station: "",
    area: "",
    phone: "",
    status: "draft" as Police["status"],
  });

  const handleAdd = () => {
    setEditingPolice(null);
    setFormData({
      name: "",
      designation: "",
      station: "",
      area: "",
      phone: "",
      status: "draft",
    });
    setIsDialogOpen(true);
  };

  const handleEdit = (item: Police) => {
    setEditingPolice(item);
    setFormData({
      name: item.name,
      designation: item.designation,
      station: item.station,
      area: item.area,
      phone: item.phone,
      status: item.status,
    });
    setIsDialogOpen(true);
  };

  const handleDelete = (item: Police) => {
    setPolice(police.filter((p) => p.id !== item.id));
    toast.success("পুলিশ তথ্য সফলভাবে মুছে ফেলা হয়েছে");
  };

  const handleSubmit = () => {
    if (editingPolice) {
      setPolice(
        police.map((p) =>
          p.id === editingPolice.id ? { ...p, ...formData } : p
        )
      );
      toast.success("পুলিশ তথ্য সফলভাবে আপডেট হয়েছে");
    } else {
      const newPolice: Police = {
        id: String(Date.now()),
        ...formData,
      };
      setPolice([newPolice, ...police]);
      toast.success("নতুন পুলিশ তথ্য সফলভাবে যুক্ত হয়েছে");
    }
    setIsDialogOpen(false);
  };

  return (
    <AdminLayout
      title="পুলিশ ম্যানেজমেন্ট"
      description="সকল পুলিশ কর্মকর্তার তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={police}
        columns={columns}
        onAdd={handleAdd}
        onEdit={handleEdit}
        onDelete={handleDelete}
        addButtonText="নতুন পুলিশ তথ্য যুক্ত করুন"
        searchPlaceholder="পুলিশের নাম বা থানা দিয়ে খুঁজুন..."
      />

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-[500px]">
          <DialogHeader>
            <DialogTitle>
              {editingPolice ? "পুলিশ তথ্য সম্পাদনা" : "নতুন পুলিশ তথ্য যুক্ত করুন"}
            </DialogTitle>
            <DialogDescription>
              পুলিশ কর্মকর্তার সকল তথ্য সঠিকভাবে পূরণ করুন
            </DialogDescription>
          </DialogHeader>

          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <Label htmlFor="name">নাম</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                placeholder="এস.আই. মোঃ রফিক"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="designation">পদবী</Label>
                <Select
                  value={formData.designation}
                  onValueChange={(value) => setFormData({ ...formData, designation: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="নির্বাচন করুন" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="অফিসার ইনচার্জ">অফিসার ইনচার্জ</SelectItem>
                    <SelectItem value="সাব-ইন্সপেক্টর">সাব-ইন্সপেক্টর</SelectItem>
                    <SelectItem value="এ.এস.আই">এ.এস.আই</SelectItem>
                    <SelectItem value="কনস্টেবল">কনস্টেবল</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="grid gap-2">
                <Label htmlFor="phone">ফোন নম্বর</Label>
                <Input
                  id="phone"
                  value={formData.phone}
                  onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                  placeholder="০১৭১২৩৪৫৬৭৮"
                />
              </div>
            </div>

            <div className="grid gap-2">
              <Label htmlFor="station">থানা</Label>
              <Input
                id="station"
                value={formData.station}
                onChange={(e) => setFormData({ ...formData, station: e.target.value })}
                placeholder="শীতলকার থানা"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="area">এলাকা</Label>
              <Input
                id="area"
                value={formData.area}
                onChange={(e) => setFormData({ ...formData, area: e.target.value })}
                placeholder="শীতলকার, বরিশাল"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="status">স্ট্যাটাস</Label>
              <Select
                value={formData.status}
                onValueChange={(value: Police["status"]) =>
                  setFormData({ ...formData, status: value })
                }
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="published">প্রকাশিত</SelectItem>
                  <SelectItem value="pending">অপেক্ষমান</SelectItem>
                  <SelectItem value="draft">খসড়া</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
              বাতিল
            </Button>
            <Button onClick={handleSubmit} className="btn-gradient-primary">
              {editingPolice ? "আপডেট করুন" : "যুক্ত করুন"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AdminLayout>
  );
}
