import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import { Player } from "@/types";
import { Phone, Gamepad2, Medal, Users } from "lucide-react";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Badge } from "@/components/ui/badge";

// Mock data
const mockPlayers: Player[] = [
  {
    id: "1",
    name: "মোঃ সাকিব হোসেন",
    position: "স্ট্রাইকার",
    playType: "ফুটবল",
    phone: "০১৭১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "2",
    name: "রাহাত আলী",
    position: "ব্যাটসম্যান",
    playType: "ক্রিকেট",
    phone: "০১৮১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "3",
    name: "জুনায়েদ আহমেদ",
    position: "গোলকিপার",
    playType: "ফুটবল",
    phone: "০১৯১২৩৪৫৬৭৮",
    status: "pending",
  },
  {
    id: "4",
    name: "ফাহিম মাহমুদ",
    position: "বোলার",
    playType: "ক্রিকেট",
    phone: "০১৫১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "5",
    name: "তানভীর হাসান",
    position: "মিডফিল্ডার",
    playType: "ফুটবল",
    phone: "০১৬১২৩৪৫৬৭৮",
    status: "draft",
  },
];

const getPlayTypeColor = (type: string) => {
  switch (type) {
    case "ফুটবল":
      return "bg-primary/10 text-primary border-primary/30";
    case "ক্রিকেট":
      return "bg-success/10 text-success border-success/30";
    case "ভলিবল":
      return "bg-warning/10 text-warning border-warning/30";
    default:
      return "bg-muted text-muted-foreground";
  }
};

const columns: Column<Player>[] = [
  {
    key: "name",
    header: "খেলোয়াড়",
    render: (item) => (
      <div className="flex items-center gap-3">
        <Avatar className="h-10 w-10">
          <AvatarImage src={item.image} />
          <AvatarFallback className="bg-accent/20 text-accent-foreground font-medium">
            {item.name.charAt(0)}
          </AvatarFallback>
        </Avatar>
        <div>
          <p className="font-medium">{item.name}</p>
          <p className="text-sm text-muted-foreground">{item.position}</p>
        </div>
      </div>
    ),
  },
  {
    key: "playType",
    header: "খেলা",
    render: (item) => (
      <Badge className={`${getPlayTypeColor(item.playType)} font-medium`}>
        <Gamepad2 className="h-3 w-3 mr-1" />
        {item.playType}
      </Badge>
    ),
  },
  {
    key: "position",
    header: "পজিশন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Medal className="h-4 w-4 text-muted-foreground" />
        <span>{item.position}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function PlayersPage() {
  const [players, setPlayers] = useState<Player[]>(mockPlayers);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingPlayer, setEditingPlayer] = useState<Player | null>(null);
  const [formData, setFormData] = useState({
    name: "",
    position: "",
    playType: "",
    phone: "",
    status: "draft" as Player["status"],
  });

  const handleAdd = () => {
    setEditingPlayer(null);
    setFormData({
      name: "",
      position: "",
      playType: "",
      phone: "",
      status: "draft",
    });
    setIsDialogOpen(true);
  };

  const handleEdit = (player: Player) => {
    setEditingPlayer(player);
    setFormData({
      name: player.name,
      position: player.position,
      playType: player.playType,
      phone: player.phone,
      status: player.status,
    });
    setIsDialogOpen(true);
  };

  const handleDelete = (player: Player) => {
    setPlayers(players.filter((p) => p.id !== player.id));
    toast.success("খেলোয়াড় সফলভাবে মুছে ফেলা হয়েছে");
  };

  const handleSubmit = () => {
    if (editingPlayer) {
      setPlayers(
        players.map((p) =>
          p.id === editingPlayer.id ? { ...p, ...formData } : p
        )
      );
      toast.success("খেলোয়াড় তথ্য সফলভাবে আপডেট হয়েছে");
    } else {
      const newPlayer: Player = {
        id: String(Date.now()),
        ...formData,
      };
      setPlayers([newPlayer, ...players]);
      toast.success("নতুন খেলোয়াড় সফলভাবে যুক্ত হয়েছে");
    }
    setIsDialogOpen(false);
  };

  return (
    <AdminLayout
      title="খেলোয়াড় ম্যানেজমেন্ট"
      description="সকল খেলোয়াড়ের তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={players}
        columns={columns}
        onAdd={handleAdd}
        onEdit={handleEdit}
        onDelete={handleDelete}
        addButtonText="নতুন খেলোয়াড় যুক্ত করুন"
        searchPlaceholder="খেলোয়াড়ের নাম বা খেলা দিয়ে খুঁজুন..."
      />

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-[500px]">
          <DialogHeader>
            <DialogTitle>
              {editingPlayer ? "খেলোয়াড় তথ্য সম্পাদনা" : "নতুন খেলোয়াড় যুক্ত করুন"}
            </DialogTitle>
            <DialogDescription>
              খেলোয়াড়ের সকল তথ্য সঠিকভাবে পূরণ করুন
            </DialogDescription>
          </DialogHeader>

          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <Label htmlFor="name">নাম</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                placeholder="মোঃ সাকিব হোসেন"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="playType">খেলা</Label>
                <Select
                  value={formData.playType}
                  onValueChange={(value) => setFormData({ ...formData, playType: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="নির্বাচন করুন" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ফুটবল">ফুটবল</SelectItem>
                    <SelectItem value="ক্রিকেট">ক্রিকেট</SelectItem>
                    <SelectItem value="ভলিবল">ভলিবল</SelectItem>
                    <SelectItem value="ব্যাডমিন্টন">ব্যাডমিন্টন</SelectItem>
                    <SelectItem value="কাবাডি">কাবাডি</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="grid gap-2">
                <Label htmlFor="position">পজিশন</Label>
                <Input
                  id="position"
                  value={formData.position}
                  onChange={(e) => setFormData({ ...formData, position: e.target.value })}
                  placeholder="স্ট্রাইকার"
                />
              </div>
            </div>

            <div className="grid gap-2">
              <Label htmlFor="phone">ফোন নম্বর</Label>
              <Input
                id="phone"
                value={formData.phone}
                onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                placeholder="০১৭১২৩৪৫৬৭৮"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="status">স্ট্যাটাস</Label>
              <Select
                value={formData.status}
                onValueChange={(value: Player["status"]) =>
                  setFormData({ ...formData, status: value })
                }
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="published">প্রকাশিত</SelectItem>
                  <SelectItem value="pending">অপেক্ষমান</SelectItem>
                  <SelectItem value="draft">খসড়া</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
              বাতিল
            </Button>
            <Button onClick={handleSubmit} className="btn-gradient-primary">
              {editingPlayer ? "আপডেট করুন" : "যুক্ত করুন"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AdminLayout>
  );
}
