import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { useState } from "react";
import { toast } from "sonner";
import { Phone, MapPin, Building2, Shield } from "lucide-react";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";

interface MemberData {
  id: string;
  name: string;
  designation: string;
  union: string;
  area: string;
  phone: string;
  imageUrl?: string;
  status: "published" | "pending" | "draft";
}

const mockMembers: MemberData[] = [
  {
    id: "1",
    name: "জনাব আব্দুল করিম",
    designation: "ইউনিয়ন চেয়ারম্যান",
    union: "শীতলকার ইউনিয়ন",
    area: "শীতলকার, বরিশাল",
    phone: "০১৭১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "2",
    name: "জনাব মোঃ রফিক",
    designation: "ইউপি মেম্বার",
    union: "শীতলকার ইউনিয়ন",
    area: "পশ্চিম পাড়া",
    phone: "০১৮১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "3",
    name: "মিসেস ফাতেমা বেগম",
    designation: "ইউপি মেম্বার (মহিলা আসন)",
    union: "শীতলকার ইউনিয়ন",
    area: "পূর্ব পাড়া",
    phone: "০১৯১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "4",
    name: "জনাব জাহিদ হাসান",
    designation: "ইউপি মেম্বার",
    union: "শীতলকার ইউনিয়ন",
    area: "উত্তর পাড়া",
    phone: "০১৫১২৩৪৫৬৭৮",
    status: "pending",
  },
];

const columns: Column<MemberData>[] = [
  {
    key: "name",
    header: "নাম",
    render: (item) => (
      <div className="flex items-center gap-3">
        <Avatar className="h-10 w-10">
          <AvatarImage src={item.imageUrl} />
          <AvatarFallback className="bg-primary/10 text-primary font-medium">
            {item.name.charAt(0)}
          </AvatarFallback>
        </Avatar>
        <div>
          <p className="font-medium">{item.name}</p>
          <div className="flex items-center gap-1 text-sm text-muted-foreground">
            <Shield className="h-3 w-3" />
            <span>{item.designation}</span>
          </div>
        </div>
      </div>
    ),
  },
  {
    key: "union",
    header: "ইউনিয়ন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Building2 className="h-4 w-4 text-muted-foreground" />
        <span>{item.union}</span>
      </div>
    ),
  },
  {
    key: "area",
    header: "এলাকা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.area}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function MembersPage() {
  const [members, setMembers] = useState<MemberData[]>(mockMembers);

  const handleDelete = (item: MemberData) => {
    setMembers(members.filter((m) => m.id !== item.id));
    toast.success("মেম্বার তথ্য সফলভাবে মুছে ফেলা হয়েছে");
  };

  return (
    <AdminLayout
      title="চেয়ারম্যান/মেম্বার ম্যানেজমেন্ট"
      description="সকল চেয়ারম্যান ও মেম্বারের তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={members}
        columns={columns}
        onAdd={() => toast.info("নতুন মেম্বার যুক্ত করার ফর্ম আসবে")}
        onEdit={(item) => toast.info(`${item.name} সম্পাদনা করা হচ্ছে`)}
        onDelete={handleDelete}
        addButtonText="নতুন মেম্বার যুক্ত করুন"
        searchPlaceholder="মেম্বারের নাম বা পদবী দিয়ে খুঁজুন..."
      />
    </AdminLayout>
  );
}
