import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { useState } from "react";
import { toast } from "sonner";
import { Phone, MapPin, Scale, Briefcase } from "lucide-react";

interface LawyerData {
  id: string;
  name: string;
  specialization: string;
  address: string;
  phone: string;
  status: "published" | "pending" | "draft";
}

const mockLawyers: LawyerData[] = [
  {
    id: "1",
    name: "অ্যাডভোকেট মোঃ সাইফুল ইসলাম",
    specialization: "ফৌজদারি মামলা",
    address: "বরিশাল জজ কোর্ট",
    phone: "০১৭১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "2",
    name: "অ্যাডভোকেট ফাতেমা বেগম",
    specialization: "পারিবারিক মামলা",
    address: "বরিশাল পারিবারিক আদালত",
    phone: "০১৮১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "3",
    name: "অ্যাডভোকেট রহিম উদ্দিন",
    specialization: "জমি সংক্রান্ত মামলা",
    address: "বানারীপাড়া",
    phone: "০১৯১২৩৪৫৬৭৮",
    status: "pending",
  },
];

const columns: Column<LawyerData>[] = [
  {
    key: "name",
    header: "আইনজীবী",
    render: (item) => (
      <div className="flex items-center gap-3">
        <div className="h-10 w-10 rounded-full bg-accent/20 flex items-center justify-center">
          <Scale className="h-5 w-5 text-accent-foreground" />
        </div>
        <div>
          <p className="font-medium">{item.name}</p>
          <div className="flex items-center gap-1 text-sm text-muted-foreground">
            <Briefcase className="h-3 w-3" />
            <span>{item.specialization}</span>
          </div>
        </div>
      </div>
    ),
  },
  {
    key: "address",
    header: "ঠিকানা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.address}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function LawyersPage() {
  const [lawyers, setLawyers] = useState<LawyerData[]>(mockLawyers);

  const handleDelete = (item: LawyerData) => {
    setLawyers(lawyers.filter((l) => l.id !== item.id));
    toast.success("আইনজীবী তথ্য সফলভাবে মুছে ফেলা হয়েছে");
  };

  return (
    <AdminLayout
      title="আইনজীবী ম্যানেজমেন্ট"
      description="সকল আইনজীবীর তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={lawyers}
        columns={columns}
        onAdd={() => toast.info("নতুন আইনজীবী যুক্ত করার ফর্ম আসবে")}
        onEdit={(item) => toast.info(`${item.name} সম্পাদনা করা হচ্ছে`)}
        onDelete={handleDelete}
        addButtonText="নতুন আইনজীবী যুক্ত করুন"
        searchPlaceholder="আইনজীবীর নাম বা বিশেষজ্ঞতা দিয়ে খুঁজুন..."
      />
    </AdminLayout>
  );
}
