import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { useState } from "react";
import { toast } from "sonner";
import { Phone, MapPin, Building2, User } from "lucide-react";

interface ImamData {
  id: string;
  name: string;
  mosque: string;
  address: string;
  phone: string;
  status: "published" | "pending" | "draft";
}

const mockImams: ImamData[] = [
  {
    id: "1",
    name: "মাওলানা আব্দুল করিম",
    mosque: "শীতলকার জামে মসজিদ",
    address: "শীতলকার বাজার, বরিশাল",
    phone: "০১৭১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "2",
    name: "হাফেজ মোঃ রফিক",
    mosque: "মধ্যবাড়ী জামে মসজিদ",
    address: "মধ্যবাড়ী, বানারীপাড়া",
    phone: "০১৮১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "3",
    name: "মাওলানা জাহিদ হাসান",
    mosque: "উজিরপুর কেন্দ্রীয় মসজিদ",
    address: "উজিরপুর সদর",
    phone: "০১৯১২৩৪৫৬৭৮",
    status: "pending",
  },
];

const columns: Column<ImamData>[] = [
  {
    key: "name",
    header: "ইমাম সাহেব",
    render: (item) => (
      <div className="flex items-center gap-3">
        <div className="h-10 w-10 rounded-full bg-primary/10 flex items-center justify-center">
          <User className="h-5 w-5 text-primary" />
        </div>
        <div>
          <p className="font-medium">{item.name}</p>
        </div>
      </div>
    ),
  },
  {
    key: "mosque",
    header: "মসজিদ",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Building2 className="h-4 w-4 text-muted-foreground" />
        <span>{item.mosque}</span>
      </div>
    ),
  },
  {
    key: "address",
    header: "ঠিকানা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.address}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function ImamsPage() {
  const [imams, setImams] = useState<ImamData[]>(mockImams);

  const handleDelete = (item: ImamData) => {
    setImams(imams.filter((i) => i.id !== item.id));
    toast.success("ইমাম সাহেবের তথ্য সফলভাবে মুছে ফেলা হয়েছে");
  };

  return (
    <AdminLayout
      title="ইমাম সাহেব ম্যানেজমেন্ট"
      description="সকল ইমাম সাহেবের তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={imams}
        columns={columns}
        onAdd={() => toast.info("নতুন ইমাম সাহেব যুক্ত করার ফর্ম আসবে")}
        onEdit={(item) => toast.info(`${item.name} সম্পাদনা করা হচ্ছে`)}
        onDelete={handleDelete}
        addButtonText="নতুন ইমাম সাহেব যুক্ত করুন"
        searchPlaceholder="ইমাম সাহেবের নাম বা মসজিদ দিয়ে খুঁজুন..."
      />
    </AdminLayout>
  );
}
