import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { useState } from "react";
import { toast } from "sonner";
import { Phone, MapPin, Flame } from "lucide-react";

interface FireServiceData {
  id: string;
  name: string;
  address: string;
  phone: string;
  status: "published" | "pending" | "draft";
}

const mockFireServices: FireServiceData[] = [
  {
    id: "1",
    name: "শীতলকার ফায়ার স্টেশন",
    address: "শীতলকার বাজার, বরিশাল",
    phone: "০১৭১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "2",
    name: "বানারীপাড়া ফায়ার সার্ভিস",
    address: "বানারীপাড়া সদর",
    phone: "০১৮১২৩৪৫৬৭৮",
    status: "published",
  },
  {
    id: "3",
    name: "বরিশাল সেন্ট্রাল ফায়ার স্টেশন",
    address: "বরিশাল সদর",
    phone: "১৯৯",
    status: "published",
  },
];

const columns: Column<FireServiceData>[] = [
  {
    key: "name",
    header: "স্টেশন",
    render: (item) => (
      <div className="flex items-center gap-3">
        <div className="h-10 w-10 rounded-full bg-destructive/10 flex items-center justify-center">
          <Flame className="h-5 w-5 text-destructive" />
        </div>
        <div>
          <p className="font-medium">{item.name}</p>
        </div>
      </div>
    ),
  },
  {
    key: "address",
    header: "ঠিকানা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.address}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span className="font-medium text-destructive">{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function FireServicePage() {
  const [fireServices, setFireServices] = useState<FireServiceData[]>(mockFireServices);

  const handleDelete = (item: FireServiceData) => {
    setFireServices(fireServices.filter((f) => f.id !== item.id));
    toast.success("ফায়ার সার্ভিস তথ্য সফলভাবে মুছে ফেলা হয়েছে");
  };

  return (
    <AdminLayout
      title="ফায়ার সার্ভিস ম্যানেজমেন্ট"
      description="সকল ফায়ার সার্ভিস স্টেশনের তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={fireServices}
        columns={columns}
        onAdd={() => toast.info("নতুন ফায়ার সার্ভিস যুক্ত করার ফর্ম আসবে")}
        onEdit={(item) => toast.info(`${item.name} সম্পাদনা করা হচ্ছে`)}
        onDelete={handleDelete}
        addButtonText="নতুন ফায়ার সার্ভিস যুক্ত করুন"
        searchPlaceholder="স্টেশন বা এলাকা দিয়ে খুঁজুন..."
      />
    </AdminLayout>
  );
}
