import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { StatCard } from "@/components/admin/StatCard";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { toast } from "sonner";
import { Donation } from "@/types";
import { Phone, DollarSign, CreditCard, Calendar, Heart, TrendingUp } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
} from "recharts";

// Mock data
const mockDonations: Donation[] = [
  {
    id: "1",
    name: "মোঃ আব্দুল করিম",
    phone: "০১৭১২৩৪৫৬৭৮",
    amount: 5000,
    method: "বিকাশ",
    transactionId: "TXN123456",
    status: "completed",
    createdAt: "২০২৫-০১-২৪",
  },
  {
    id: "2",
    name: "ফাতেমা বেগম",
    phone: "০১৮১২৩৪৫৬৭৮",
    amount: 2000,
    method: "নগদ",
    transactionId: "TXN123457",
    status: "completed",
    createdAt: "২০২৫-০১-২৩",
  },
  {
    id: "3",
    name: "রহিম উদ্দিন",
    phone: "০১৯১২৩৪৫৬৭৮",
    amount: 10000,
    method: "রকেট",
    transactionId: "",
    status: "pending",
    createdAt: "২০২৫-০১-২২",
  },
  {
    id: "4",
    name: "সালমা খাতুন",
    phone: "০১৫১২৩৪৫৬৭৮",
    amount: 3000,
    method: "বিকাশ",
    transactionId: "TXN123458",
    status: "completed",
    createdAt: "২০২৫-০১-২০",
  },
  {
    id: "5",
    name: "জাহিদ হাসান",
    phone: "০১৬১২৩৪৫৬৭৮",
    amount: 1000,
    method: "নগদ",
    transactionId: "",
    status: "failed",
    createdAt: "২০২৫-০১-১৮",
  },
];

const monthlyData = [
  { month: "জুলাই", amount: 25000 },
  { month: "আগস্ট", amount: 32000 },
  { month: "সেপ্টেম্বর", amount: 28000 },
  { month: "অক্টোবর", amount: 45000 },
  { month: "নভেম্বর", amount: 38000 },
  { month: "ডিসেম্বর", amount: 52000 },
];

const getStatusBadge = (status: Donation["status"]) => {
  switch (status) {
    case "completed":
      return <Badge className="badge-success">সম্পন্ন</Badge>;
    case "pending":
      return <Badge className="badge-warning">অপেক্ষমান</Badge>;
    case "failed":
      return <Badge className="badge-error">ব্যর্থ</Badge>;
  }
};

const columns: Column<Donation>[] = [
  {
    key: "name",
    header: "দাতা",
    render: (item) => (
      <div>
        <p className="font-medium">{item.name}</p>
        <div className="flex items-center gap-1 text-sm text-muted-foreground">
          <Phone className="h-3 w-3" />
          <span>{item.phone}</span>
        </div>
      </div>
    ),
  },
  {
    key: "amount",
    header: "পরিমাণ",
    render: (item) => (
      <div className="font-bold text-lg text-primary">
        ৳{item.amount.toLocaleString("bn-BD")}
      </div>
    ),
  },
  {
    key: "method",
    header: "পেমেন্ট মাধ্যম",
    render: (item) => (
      <div className="flex items-center gap-2">
        <CreditCard className="h-4 w-4 text-muted-foreground" />
        <span>{item.method}</span>
      </div>
    ),
  },
  {
    key: "transactionId",
    header: "ট্রানজেকশন আইডি",
    render: (item) => (
      <span className="font-mono text-sm">
        {item.transactionId || "-"}
      </span>
    ),
  },
  {
    key: "createdAt",
    header: "তারিখ",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Calendar className="h-4 w-4 text-muted-foreground" />
        <span>{item.createdAt}</span>
      </div>
    ),
  },
  {
    key: "status",
    header: "স্ট্যাটাস",
    render: (item) => getStatusBadge(item.status),
  },
];

export default function DonationsPage() {
  const [donations] = useState<Donation[]>(mockDonations);

  const totalAmount = donations
    .filter((d) => d.status === "completed")
    .reduce((sum, d) => sum + d.amount, 0);
  const pendingAmount = donations
    .filter((d) => d.status === "pending")
    .reduce((sum, d) => sum + d.amount, 0);
  const totalDonors = new Set(donations.map((d) => d.phone)).size;

  return (
    <AdminLayout
      title="ডোনেশন ম্যানেজমেন্ট"
      description="সকল ডোনেশন দেখুন এবং পরিচালনা করুন"
    >
      {/* Stats */}
      <div className="grid gap-4 md:gap-6 grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 mb-6 md:mb-8">
        <StatCard
          title="মোট ডোনেশন"
          value={`৳${totalAmount.toLocaleString("bn-BD")}`}
          icon={Heart}
          variant="secondary"
          trend={{ value: 15, isPositive: true }}
        />
        <StatCard
          title="অপেক্ষমান"
          value={`৳${pendingAmount.toLocaleString("bn-BD")}`}
          icon={DollarSign}
          variant="warning"
        />
        <StatCard
          title="মোট দাতা"
          value={totalDonors}
          icon={TrendingUp}
          variant="primary"
        />
        <StatCard
          title="এই মাসে"
          value={`৳${(52000).toLocaleString("bn-BD")}`}
          icon={Calendar}
          variant="success"
        />
      </div>

      {/* Chart */}
      <Card className="glass-card mb-6">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="h-5 w-5 text-primary" />
            মাসিক ডোনেশন
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="h-[300px]">
            <ResponsiveContainer width="100%" height="100%">
              <BarChart data={monthlyData}>
                <CartesianGrid strokeDasharray="3 3" stroke="hsl(220, 13%, 91%)" />
                <XAxis dataKey="month" stroke="hsl(220, 9%, 46%)" fontSize={12} />
                <YAxis stroke="hsl(220, 9%, 46%)" fontSize={12} />
                <Tooltip
                  contentStyle={{
                    backgroundColor: "hsl(0, 0%, 100%)",
                    border: "1px solid hsl(220, 13%, 91%)",
                    borderRadius: "8px",
                  }}
                  formatter={(value: number) => [`৳${value.toLocaleString("bn-BD")}`, "পরিমাণ"]}
                />
                <Bar
                  dataKey="amount"
                  fill="hsl(152, 69%, 31%)"
                  radius={[4, 4, 0, 0]}
                />
              </BarChart>
            </ResponsiveContainer>
          </div>
        </CardContent>
      </Card>

      {/* Table */}
      <DataTable
        data={donations}
        columns={columns}
        showActions={false}
        searchPlaceholder="দাতার নাম বা ফোন দিয়ে খুঁজুন..."
      />
    </AdminLayout>
  );
}
