import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import { Doctor } from "@/types";
import { Phone, MapPin, Building2 } from "lucide-react";

// Mock data
const mockDoctors: Doctor[] = [
  {
    id: "1",
    name: "ডাঃ মোঃ আব্দুল করিম",
    specialization: "মেডিসিন বিশেষজ্ঞ",
    hospital: "শীতলকার সরকারি হাসপাতাল",
    area: "শীতলকার, বরিশাল",
    phone: "০১৭১২৩৪৫৬৭৮",
    status: "published",
    createdAt: "২০২৫-০১-১০",
    updatedAt: "২০২৫-০১-১০",
  },
  {
    id: "2",
    name: "ডাঃ ফাতেমা বেগম",
    specialization: "গাইনি বিশেষজ্ঞ",
    hospital: "মাতৃসেবা ক্লিনিক",
    area: "কালীবাড়ী, বরিশাল",
    phone: "০১৮১২৩৪৫৬৭৮",
    status: "published",
    createdAt: "২০২৫-০১-০৮",
    updatedAt: "২০২৫-০১-০৮",
  },
  {
    id: "3",
    name: "ডাঃ রহিম উদ্দিন",
    specialization: "শিশু বিশেষজ্ঞ",
    hospital: "শিশু স্বাস্থ্য কেন্দ্র",
    area: "বানারীপাড়া",
    phone: "০১৯১২৩৪৫৬৭৮",
    status: "pending",
    createdAt: "২০২৫-০১-১৫",
    updatedAt: "২০২৫-০১-১৫",
  },
  {
    id: "4",
    name: "ডাঃ সালমা খাতুন",
    specialization: "চক্ষু বিশেষজ্ঞ",
    hospital: "আই কেয়ার সেন্টার",
    area: "উজিরপুর",
    phone: "০১৫১২৩৪৫৬৭৮",
    status: "published",
    createdAt: "২০২৫-০১-০৫",
    updatedAt: "২০২৫-০১-০৫",
  },
  {
    id: "5",
    name: "ডাঃ জাহিদ হাসান",
    specialization: "হৃদরোগ বিশেষজ্ঞ",
    hospital: "হার্ট কেয়ার হাসপাতাল",
    area: "বরিশাল সদর",
    phone: "০১৬১২৩৪৫৬৭৮",
    status: "draft",
    createdAt: "২০২৫-০১-২০",
    updatedAt: "২০২৫-০১-২০",
  },
];

const columns: Column<Doctor>[] = [
  {
    key: "name",
    header: "নাম",
    render: (item) => (
      <div>
        <p className="font-medium">{item.name}</p>
        <p className="text-sm text-muted-foreground">{item.specialization}</p>
      </div>
    ),
  },
  {
    key: "hospital",
    header: "হাসপাতাল",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Building2 className="h-4 w-4 text-muted-foreground" />
        <span>{item.hospital}</span>
      </div>
    ),
  },
  {
    key: "area",
    header: "এলাকা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.area}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function DoctorsPage() {
  const [doctors, setDoctors] = useState<Doctor[]>(mockDoctors);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingDoctor, setEditingDoctor] = useState<Doctor | null>(null);
  const [formData, setFormData] = useState({
    name: "",
    specialization: "",
    hospital: "",
    area: "",
    phone: "",
    status: "draft" as Doctor["status"],
  });

  const handleAdd = () => {
    setEditingDoctor(null);
    setFormData({
      name: "",
      specialization: "",
      hospital: "",
      area: "",
      phone: "",
      status: "draft",
    });
    setIsDialogOpen(true);
  };

  const handleEdit = (doctor: Doctor) => {
    setEditingDoctor(doctor);
    setFormData({
      name: doctor.name,
      specialization: doctor.specialization,
      hospital: doctor.hospital,
      area: doctor.area,
      phone: doctor.phone,
      status: doctor.status,
    });
    setIsDialogOpen(true);
  };

  const handleDelete = (doctor: Doctor) => {
    setDoctors(doctors.filter((d) => d.id !== doctor.id));
    toast.success("ডক্টর সফলভাবে মুছে ফেলা হয়েছে");
  };

  const handleSubmit = () => {
    if (editingDoctor) {
      setDoctors(
        doctors.map((d) =>
          d.id === editingDoctor.id
            ? { ...d, ...formData, updatedAt: new Date().toLocaleDateString("bn-BD") }
            : d
        )
      );
      toast.success("ডক্টর তথ্য সফলভাবে আপডেট হয়েছে");
    } else {
      const newDoctor: Doctor = {
        id: String(Date.now()),
        ...formData,
        createdAt: new Date().toLocaleDateString("bn-BD"),
        updatedAt: new Date().toLocaleDateString("bn-BD"),
      };
      setDoctors([newDoctor, ...doctors]);
      toast.success("নতুন ডক্টর সফলভাবে যুক্ত হয়েছে");
    }
    setIsDialogOpen(false);
  };

  return (
    <AdminLayout
      title="ডক্টর ম্যানেজমেন্ট"
      description="সকল ডক্টরের তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={doctors}
        columns={columns}
        onAdd={handleAdd}
        onEdit={handleEdit}
        onDelete={handleDelete}
        onView={(item) => console.log("View", item)}
        addButtonText="নতুন ডক্টর যুক্ত করুন"
        searchPlaceholder="ডক্টরের নাম বা বিশেষজ্ঞতা দিয়ে খুঁজুন..."
      />

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-[500px]">
          <DialogHeader>
            <DialogTitle>
              {editingDoctor ? "ডক্টর তথ্য সম্পাদনা" : "নতুন ডক্টর যুক্ত করুন"}
            </DialogTitle>
            <DialogDescription>
              ডক্টরের সকল তথ্য সঠিকভাবে পূরণ করুন
            </DialogDescription>
          </DialogHeader>

          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <Label htmlFor="name">নাম</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                placeholder="ডাঃ মোঃ আব্দুল করিম"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="specialization">বিশেষজ্ঞতা</Label>
              <Input
                id="specialization"
                value={formData.specialization}
                onChange={(e) =>
                  setFormData({ ...formData, specialization: e.target.value })
                }
                placeholder="মেডিসিন বিশেষজ্ঞ"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="hospital">হাসপাতাল/ক্লিনিক</Label>
              <Input
                id="hospital"
                value={formData.hospital}
                onChange={(e) => setFormData({ ...formData, hospital: e.target.value })}
                placeholder="শীতলকার সরকারি হাসপাতাল"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="area">এলাকা</Label>
                <Input
                  id="area"
                  value={formData.area}
                  onChange={(e) => setFormData({ ...formData, area: e.target.value })}
                  placeholder="শীতলকার"
                />
              </div>

              <div className="grid gap-2">
                <Label htmlFor="phone">ফোন নম্বর</Label>
                <Input
                  id="phone"
                  value={formData.phone}
                  onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                  placeholder="০১৭১২৩৪৫৬৭৮"
                />
              </div>
            </div>

            <div className="grid gap-2">
              <Label htmlFor="status">স্ট্যাটাস</Label>
              <Select
                value={formData.status}
                onValueChange={(value: Doctor["status"]) =>
                  setFormData({ ...formData, status: value })
                }
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="published">প্রকাশিত</SelectItem>
                  <SelectItem value="pending">অপেক্ষমান</SelectItem>
                  <SelectItem value="draft">খসড়া</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
              বাতিল
            </Button>
            <Button onClick={handleSubmit} className="btn-gradient-primary">
              {editingDoctor ? "আপডেট করুন" : "যুক্ত করুন"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AdminLayout>
  );
}
