import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { StatCard } from "@/components/admin/StatCard";
import { DataTable, Column } from "@/components/admin/DataTable";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Stethoscope,
  Droplets,
  Users,
  Gamepad2,
  TrendingUp,
  Clock,
  UserCheck,
  DollarSign,
  Activity,
  Calendar,
  MapPin,
  Phone,
} from "lucide-react";
import {
  AreaChart,
  Area,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  BarChart,
  Bar,
  PieChart,
  Pie,
  Cell,
} from "recharts";

// Mock data for charts
const visitorsData = [
  { name: "শনি", visitors: 400 },
  { name: "রবি", visitors: 600 },
  { name: "সোম", visitors: 800 },
  { name: "মঙ্গল", visitors: 750 },
  { name: "বুধ", visitors: 900 },
  { name: "বৃহস্পতি", visitors: 650 },
  { name: "শুক্র", visitors: 500 },
];

const serviceData = [
  { name: "ডক্টর", count: 145, color: "#16a34a" },
  { name: "রক্তদাতা", count: 892, color: "#dc2626" },
  { name: "পুলিশ", count: 78, color: "#2563eb" },
  { name: "অ্যাম্বুলেন্স", count: 34, color: "#f59e0b" },
  { name: "আইনজীবী", count: 56, color: "#8b5cf6" },
];

const recentActivities = [
  { id: "1", type: "blood", message: "নতুন রক্তদাতা অনুরোধ - আব্দুল করিম", time: "৫ মিনিট আগে" },
  { id: "2", type: "club", message: "ক্লাব সদস্য অনুমোদন অপেক্ষমান", time: "১৫ মিনিট আগে" },
  { id: "3", type: "donation", message: "৫০০০ টাকা ডোনেশন পাওয়া গেছে", time: "৩০ মিনিট আগে" },
  { id: "4", type: "doctor", message: "নতুন ডক্টর যুক্ত করা হয়েছে", time: "১ ঘণ্টা আগে" },
  { id: "5", type: "player", message: "খেলোয়াড় তথ্য আপডেট হয়েছে", time: "২ ঘণ্টা আগে" },
];

const pendingRequests = [
  { id: "1", name: "মোঃ রফিকুল ইসলাম", type: "রক্তদাতা", phone: "০১৭১২৩৪৫৬৭৮", status: "pending" },
  { id: "2", name: "ফাতেমা আক্তার", type: "ক্লাব সদস্য", phone: "০১৮১২৩৪৫৬৭৮", status: "pending" },
  { id: "3", name: "করিম উদ্দিন", type: "রক্তদাতা", phone: "০১৯১২৩৪৫৬৭৮", status: "pending" },
];

const pendingColumns: Column<typeof pendingRequests[0]>[] = [
  { key: "name", header: "নাম" },
  { key: "type", header: "ধরন" },
  { key: "phone", header: "ফোন" },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function Dashboard() {
  return (
    <AdminLayout title="ড্যাশবোর্ড" description="শীতলকার কমিউনিটি সার্ভিস অ্যাডমিন প্যানেল">
      {/* Stats Grid */}
      <div className="grid gap-4 md:gap-6 grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 mb-6 md:mb-8">
        <StatCard
          title="মোট ডক্টর"
          value="১৪৫"
          icon={Stethoscope}
          variant="primary"
          trend={{ value: 12, isPositive: true }}
        />
        <StatCard
          title="রক্তদাতা"
          value="৮৯২"
          icon={Droplets}
          variant="secondary"
          trend={{ value: 8, isPositive: true }}
        />
        <StatCard
          title="ক্লাব সদস্য"
          value="২৫৬"
          icon={Users}
          variant="success"
          trend={{ value: 5, isPositive: true }}
        />
        <StatCard
          title="অপেক্ষমান অনুরোধ"
          value="১৫"
          icon={Clock}
          variant="warning"
        />
      </div>

      {/* Charts Section */}
      <div className="grid gap-4 md:gap-6 lg:grid-cols-3 mb-6 md:mb-8">
        {/* Visitors Chart */}
        <Card className="lg:col-span-2 glass-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="h-5 w-5 text-primary" />
              সাপ্তাহিক ভিজিটর
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="h-[300px]">
              <ResponsiveContainer width="100%" height="100%">
                <AreaChart data={visitorsData}>
                  <defs>
                    <linearGradient id="colorVisitors" x1="0" y1="0" x2="0" y2="1">
                      <stop offset="5%" stopColor="hsl(152, 69%, 31%)" stopOpacity={0.3} />
                      <stop offset="95%" stopColor="hsl(152, 69%, 31%)" stopOpacity={0} />
                    </linearGradient>
                  </defs>
                  <CartesianGrid strokeDasharray="3 3" stroke="hsl(220, 13%, 91%)" />
                  <XAxis dataKey="name" stroke="hsl(220, 9%, 46%)" fontSize={12} />
                  <YAxis stroke="hsl(220, 9%, 46%)" fontSize={12} />
                  <Tooltip
                    contentStyle={{
                      backgroundColor: "hsl(0, 0%, 100%)",
                      border: "1px solid hsl(220, 13%, 91%)",
                      borderRadius: "8px",
                    }}
                  />
                  <Area
                    type="monotone"
                    dataKey="visitors"
                    stroke="hsl(152, 69%, 31%)"
                    strokeWidth={2}
                    fillOpacity={1}
                    fill="url(#colorVisitors)"
                  />
                </AreaChart>
              </ResponsiveContainer>
            </div>
          </CardContent>
        </Card>

        {/* Service Distribution */}
        <Card className="glass-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="h-5 w-5 text-primary" />
              সার্ভিস বিতরণ
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="h-[200px]">
              <ResponsiveContainer width="100%" height="100%">
                <PieChart>
                  <Pie
                    data={serviceData}
                    cx="50%"
                    cy="50%"
                    innerRadius={50}
                    outerRadius={80}
                    paddingAngle={5}
                    dataKey="count"
                  >
                    {serviceData.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={entry.color} />
                    ))}
                  </Pie>
                  <Tooltip />
                </PieChart>
              </ResponsiveContainer>
            </div>
            <div className="mt-4 space-y-2">
              {serviceData.map((item) => (
                <div key={item.name} className="flex items-center justify-between text-sm">
                  <div className="flex items-center gap-2">
                    <div
                      className="h-3 w-3 rounded-full"
                      style={{ backgroundColor: item.color }}
                    />
                    <span>{item.name}</span>
                  </div>
                  <span className="font-medium">{item.count}</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Bottom Section */}
      <div className="grid gap-4 md:gap-6 lg:grid-cols-2">
        {/* Recent Activities */}
        <Card className="glass-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Calendar className="h-5 w-5 text-primary" />
              সাম্প্রতিক কার্যক্রম
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {recentActivities.map((activity) => (
                <div
                  key={activity.id}
                  className="flex items-start gap-3 pb-4 border-b border-border last:border-0 last:pb-0"
                >
                  <div
                    className={`h-2 w-2 mt-2 rounded-full ${
                      activity.type === "blood"
                        ? "bg-secondary"
                        : activity.type === "donation"
                        ? "bg-success"
                        : "bg-primary"
                    }`}
                  />
                  <div className="flex-1 min-w-0">
                    <p className="text-sm font-medium text-foreground">{activity.message}</p>
                    <p className="text-xs text-muted-foreground">{activity.time}</p>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Pending Requests */}
        <Card className="glass-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Clock className="h-5 w-5 text-warning" />
              অপেক্ষমান অনুরোধ
            </CardTitle>
          </CardHeader>
          <CardContent>
            <DataTable
              data={pendingRequests}
              columns={pendingColumns}
              showActions={true}
              onApprove={(item) => console.log("Approve", item)}
              onReject={(item) => console.log("Reject", item)}
            />
          </CardContent>
        </Card>
      </div>
    </AdminLayout>
  );
}
