import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import { ClubUser } from "@/types";
import { Phone, MapPin, Mail, Calendar, User } from "lucide-react";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Badge } from "@/components/ui/badge";

// Mock data
const mockClubUsers: ClubUser[] = [
  {
    id: "1",
    name: "মোঃ আরিফুল ইসলাম",
    email: "ariful@example.com",
    phone: "০১৭১২৩৪৫৬৭৮",
    dob: "১৯৯৫-০৫-১৫",
    address: "শীতলকার, বরিশাল",
    position: "সাধারণ সদস্য",
    status: "approved",
    createdAt: "২০২৫-০১-১০",
  },
  {
    id: "2",
    name: "ফাতেমা খাতুন",
    email: "fatema@example.com",
    phone: "০১৮১২৩৪৫৬৭৮",
    dob: "১৯৯৮-০৮-২০",
    address: "কালীবাড়ী, বরিশাল",
    position: "সাংস্কৃতিক সম্পাদক",
    status: "approved",
    createdAt: "২০২৫-০১-০৮",
  },
  {
    id: "3",
    name: "রাহাত হোসেন",
    email: "rahat@example.com",
    phone: "০১৯১২৩৪৫৬৭৮",
    dob: "২০০০-০১-১০",
    address: "বানারীপাড়া",
    position: "সাধারণ সদস্য",
    status: "pending",
    createdAt: "২০২৫-০১-১৫",
  },
  {
    id: "4",
    name: "সাবরিনা আক্তার",
    email: "sabrina@example.com",
    phone: "০১৫১২৩৪৫৬৭৮",
    dob: "১৯৯৭-১১-২৫",
    address: "উজিরপুর",
    position: "ক্রীড়া সম্পাদক",
    status: "approved",
    createdAt: "২০২৫-০১-০৫",
  },
  {
    id: "5",
    name: "জুনায়েদ আহমেদ",
    email: "junaid@example.com",
    phone: "০১৬১২৩৪৫৬৭৮",
    dob: "১৯৯৯-০৩-১৮",
    address: "বরিশাল সদর",
    position: "সাধারণ সদস্য",
    status: "pending",
    createdAt: "২০২৫-০১-২০",
  },
  {
    id: "6",
    name: "নুসরাত জাহান",
    email: "nusrat@example.com",
    phone: "০১৩১২৩৪৫৬৭৮",
    dob: "২০০১-০৭-০৫",
    address: "গৌরনদী",
    position: "সাধারণ সদস্য",
    status: "rejected",
    createdAt: "২০২৫-০১-১২",
  },
];

const getStatusBadge = (status: ClubUser["status"]) => {
  switch (status) {
    case "approved":
      return <Badge className="badge-success">অনুমোদিত</Badge>;
    case "pending":
      return <Badge className="badge-warning">অপেক্ষমান</Badge>;
    case "rejected":
      return <Badge className="badge-error">বাতিল</Badge>;
  }
};

const columns: Column<ClubUser>[] = [
  {
    key: "name",
    header: "সদস্য",
    render: (item) => (
      <div className="flex items-center gap-3">
        <Avatar className="h-10 w-10">
          <AvatarImage src={item.imagePath} />
          <AvatarFallback className="bg-primary/10 text-primary font-medium">
            {item.name.charAt(0)}
          </AvatarFallback>
        </Avatar>
        <div>
          <p className="font-medium">{item.name}</p>
          <p className="text-sm text-muted-foreground">{item.email}</p>
        </div>
      </div>
    ),
  },
  {
    key: "position",
    header: "পদবী",
    render: (item) => (
      <div className="flex items-center gap-2">
        <User className="h-4 w-4 text-muted-foreground" />
        <span>{item.position}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  {
    key: "address",
    header: "ঠিকানা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.address}</span>
      </div>
    ),
  },
  {
    key: "status",
    header: "স্ট্যাটাস",
    render: (item) => getStatusBadge(item.status),
  },
];

export default function ClubUsersPage() {
  const [users, setUsers] = useState<ClubUser[]>(mockClubUsers);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingUser, setEditingUser] = useState<ClubUser | null>(null);
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    dob: "",
    address: "",
    position: "",
    status: "pending" as ClubUser["status"],
  });

  const handleAdd = () => {
    setEditingUser(null);
    setFormData({
      name: "",
      email: "",
      phone: "",
      dob: "",
      address: "",
      position: "সাধারণ সদস্য",
      status: "pending",
    });
    setIsDialogOpen(true);
  };

  const handleEdit = (user: ClubUser) => {
    setEditingUser(user);
    setFormData({
      name: user.name,
      email: user.email,
      phone: user.phone,
      dob: user.dob,
      address: user.address,
      position: user.position,
      status: user.status,
    });
    setIsDialogOpen(true);
  };

  const handleDelete = (user: ClubUser) => {
    setUsers(users.filter((u) => u.id !== user.id));
    toast.success("সদস্য সফলভাবে মুছে ফেলা হয়েছে");
  };

  const handleApprove = (user: ClubUser) => {
    setUsers(
      users.map((u) =>
        u.id === user.id ? { ...u, status: "approved" as const } : u
      )
    );
    toast.success("সদস্য সফলভাবে অনুমোদিত হয়েছে");
  };

  const handleReject = (user: ClubUser) => {
    setUsers(
      users.map((u) => (u.id === user.id ? { ...u, status: "rejected" as const } : u))
    );
    toast.error("সদস্য বাতিল করা হয়েছে");
  };

  const handleSubmit = () => {
    if (editingUser) {
      setUsers(
        users.map((u) =>
          u.id === editingUser.id ? { ...u, ...formData } : u
        )
      );
      toast.success("সদস্য তথ্য সফলভাবে আপডেট হয়েছে");
    } else {
      const newUser: ClubUser = {
        id: String(Date.now()),
        ...formData,
        createdAt: new Date().toLocaleDateString("bn-BD"),
      };
      setUsers([newUser, ...users]);
      toast.success("নতুন সদস্য সফলভাবে যুক্ত হয়েছে");
    }
    setIsDialogOpen(false);
  };

  const totalUsers = users.length;
  const pendingUsers = users.filter((u) => u.status === "pending").length;
  const approvedUsers = users.filter((u) => u.status === "approved").length;

  return (
    <AdminLayout
      title="ক্লাব সদস্য ম্যানেজমেন্ট"
      description={`মোট ${totalUsers} জন | অনুমোদিত ${approvedUsers} জন | অপেক্ষমান ${pendingUsers} জন`}
    >
      <DataTable
        data={users}
        columns={columns}
        onAdd={handleAdd}
        onEdit={handleEdit}
        onDelete={handleDelete}
        onApprove={handleApprove}
        onReject={handleReject}
        addButtonText="নতুন সদস্য যুক্ত করুন"
        searchPlaceholder="সদস্যের নাম, ইমেইল বা ফোন দিয়ে খুঁজুন..."
      />

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-[500px]">
          <DialogHeader>
            <DialogTitle>
              {editingUser ? "সদস্য তথ্য সম্পাদনা" : "নতুন সদস্য যুক্ত করুন"}
            </DialogTitle>
            <DialogDescription>
              সদস্যের সকল তথ্য সঠিকভাবে পূরণ করুন
            </DialogDescription>
          </DialogHeader>

          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <Label htmlFor="name">নাম</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                placeholder="মোঃ আরিফুল ইসলাম"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="email">ইমেইল</Label>
                <Input
                  id="email"
                  type="email"
                  value={formData.email}
                  onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                  placeholder="ariful@example.com"
                />
              </div>

              <div className="grid gap-2">
                <Label htmlFor="phone">ফোন নম্বর</Label>
                <Input
                  id="phone"
                  value={formData.phone}
                  onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                  placeholder="০১৭১২৩৪৫৬৭৮"
                />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="dob">জন্ম তারিখ</Label>
                <Input
                  id="dob"
                  value={formData.dob}
                  onChange={(e) => setFormData({ ...formData, dob: e.target.value })}
                  placeholder="১৯৯৫-০৫-১৫"
                />
              </div>

              <div className="grid gap-2">
                <Label htmlFor="position">পদবী</Label>
                <Select
                  value={formData.position}
                  onValueChange={(value) => setFormData({ ...formData, position: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="নির্বাচন করুন" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="সাধারণ সদস্য">সাধারণ সদস্য</SelectItem>
                    <SelectItem value="সভাপতি">সভাপতি</SelectItem>
                    <SelectItem value="সহ-সভাপতি">সহ-সভাপতি</SelectItem>
                    <SelectItem value="সাধারণ সম্পাদক">সাধারণ সম্পাদক</SelectItem>
                    <SelectItem value="সাংস্কৃতিক সম্পাদক">সাংস্কৃতিক সম্পাদক</SelectItem>
                    <SelectItem value="ক্রীড়া সম্পাদক">ক্রীড়া সম্পাদক</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="grid gap-2">
              <Label htmlFor="address">ঠিকানা</Label>
              <Input
                id="address"
                value={formData.address}
                onChange={(e) => setFormData({ ...formData, address: e.target.value })}
                placeholder="শীতলকার, বরিশাল"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="status">স্ট্যাটাস</Label>
              <Select
                value={formData.status}
                onValueChange={(value: ClubUser["status"]) =>
                  setFormData({ ...formData, status: value })
                }
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="approved">অনুমোদিত</SelectItem>
                  <SelectItem value="pending">অপেক্ষমান</SelectItem>
                  <SelectItem value="rejected">বাতিল</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
              বাতিল
            </Button>
            <Button onClick={handleSubmit} className="btn-gradient-primary">
              {editingUser ? "আপডেট করুন" : "যুক্ত করুন"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AdminLayout>
  );
}
