import { useState } from "react";
import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import { BloodDonor } from "@/types";
import { Phone, MapPin, Droplets, Calendar } from "lucide-react";
import { Badge } from "@/components/ui/badge";

const bloodGroups = ["A+", "A-", "B+", "B-", "AB+", "AB-", "O+", "O-"];

// Mock data
const mockBloodDonors: BloodDonor[] = [
  {
    id: "1",
    name: "মোঃ রফিকুল ইসলাম",
    bloodGroup: "A+",
    phone: "০১৭১২৩৪৫৬৭৮",
    address: "শীতলকার, বরিশাল",
    hospital: "শীতলকার সরকারি হাসপাতাল",
    lastDonation: "২০২৪-১২-১৫",
    status: "published",
    createdAt: "২০২৫-০১-১০",
  },
  {
    id: "2",
    name: "ফাতেমা আক্তার",
    bloodGroup: "O+",
    phone: "০১৮১২৩৪৫৬৭৮",
    address: "কালীবাড়ী, বরিশাল",
    hospital: "বরিশাল মেডিকেল",
    lastDonation: "২০২৪-১১-২০",
    status: "published",
    createdAt: "২০২৫-০১-০৮",
  },
  {
    id: "3",
    name: "আব্দুল করিম",
    bloodGroup: "B+",
    phone: "০১৯১২৩৪৫৬৭৮",
    address: "বানারীপাড়া",
    hospital: "",
    status: "pending",
    createdAt: "২০২৫-০১-১৫",
  },
  {
    id: "4",
    name: "সাবিনা ইয়াসমিন",
    bloodGroup: "AB-",
    phone: "০১৫১২৩৪৫৬৭৮",
    address: "উজিরপুর",
    hospital: "উজিরপুর হেলথ সেন্টার",
    lastDonation: "২০২৪-১০-০৫",
    status: "published",
    createdAt: "২০২৫-০১-০৫",
  },
  {
    id: "5",
    name: "জাহিদুল ইসলাম",
    bloodGroup: "O-",
    phone: "০১৬১২৩৪৫৬৭৮",
    address: "বরিশাল সদর",
    hospital: "",
    status: "pending",
    createdAt: "২০২৫-০১-২০",
  },
];

const getBloodGroupColor = (group: string) => {
  if (group.includes("-")) return "bg-secondary/10 text-secondary border-secondary/30";
  return "bg-destructive/10 text-destructive border-destructive/30";
};

const columns: Column<BloodDonor>[] = [
  {
    key: "name",
    header: "নাম",
    render: (item) => (
      <div>
        <p className="font-medium">{item.name}</p>
        <p className="text-sm text-muted-foreground">{item.address}</p>
      </div>
    ),
  },
  {
    key: "bloodGroup",
    header: "রক্তের গ্রুপ",
    render: (item) => (
      <Badge className={`${getBloodGroupColor(item.bloodGroup)} font-bold text-base px-3 py-1`}>
        <Droplets className="h-4 w-4 mr-1" />
        {item.bloodGroup}
      </Badge>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  {
    key: "lastDonation",
    header: "সর্বশেষ ডোনেশন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Calendar className="h-4 w-4 text-muted-foreground" />
        <span>{item.lastDonation || "নেই"}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function BloodDonorsPage() {
  const [donors, setDonors] = useState<BloodDonor[]>(mockBloodDonors);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingDonor, setEditingDonor] = useState<BloodDonor | null>(null);
  const [formData, setFormData] = useState({
    name: "",
    bloodGroup: "",
    phone: "",
    address: "",
    hospital: "",
    status: "pending" as BloodDonor["status"],
  });

  const handleAdd = () => {
    setEditingDonor(null);
    setFormData({
      name: "",
      bloodGroup: "",
      phone: "",
      address: "",
      hospital: "",
      status: "pending",
    });
    setIsDialogOpen(true);
  };

  const handleEdit = (donor: BloodDonor) => {
    setEditingDonor(donor);
    setFormData({
      name: donor.name,
      bloodGroup: donor.bloodGroup,
      phone: donor.phone,
      address: donor.address,
      hospital: donor.hospital,
      status: donor.status,
    });
    setIsDialogOpen(true);
  };

  const handleDelete = (donor: BloodDonor) => {
    setDonors(donors.filter((d) => d.id !== donor.id));
    toast.success("রক্তদাতা সফলভাবে মুছে ফেলা হয়েছে");
  };

  const handleApprove = (donor: BloodDonor) => {
    setDonors(
      donors.map((d) =>
        d.id === donor.id ? { ...d, status: "published" as const } : d
      )
    );
    toast.success("রক্তদাতা সফলভাবে অনুমোদিত হয়েছে");
  };

  const handleReject = (donor: BloodDonor) => {
    setDonors(
      donors.map((d) => (d.id === donor.id ? { ...d, status: "draft" as const } : d))
    );
    toast.error("রক্তদাতা বাতিল করা হয়েছে");
  };

  const handleSubmit = () => {
    if (editingDonor) {
      setDonors(
        donors.map((d) =>
          d.id === editingDonor.id ? { ...d, ...formData } : d
        )
      );
      toast.success("রক্তদাতা তথ্য সফলভাবে আপডেট হয়েছে");
    } else {
      const newDonor: BloodDonor = {
        id: String(Date.now()),
        ...formData,
        createdAt: new Date().toLocaleDateString("bn-BD"),
      };
      setDonors([newDonor, ...donors]);
      toast.success("নতুন রক্তদাতা সফলভাবে যুক্ত হয়েছে");
    }
    setIsDialogOpen(false);
  };

  // Stats
  const totalDonors = donors.length;
  const pendingDonors = donors.filter((d) => d.status === "pending").length;

  return (
    <AdminLayout
      title="রক্তদাতা ম্যানেজমেন্ট"
      description={`মোট ${totalDonors} জন রক্তদাতা | ${pendingDonors} জন অপেক্ষমান`}
    >
      <DataTable
        data={donors}
        columns={columns}
        onAdd={handleAdd}
        onEdit={handleEdit}
        onDelete={handleDelete}
        onApprove={handleApprove}
        onReject={handleReject}
        addButtonText="নতুন রক্তদাতা যুক্ত করুন"
        searchPlaceholder="রক্তদাতার নাম বা রক্তের গ্রুপ দিয়ে খুঁজুন..."
      />

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-[500px]">
          <DialogHeader>
            <DialogTitle>
              {editingDonor ? "রক্তদাতা তথ্য সম্পাদনা" : "নতুন রক্তদাতা যুক্ত করুন"}
            </DialogTitle>
            <DialogDescription>
              রক্তদাতার সকল তথ্য সঠিকভাবে পূরণ করুন
            </DialogDescription>
          </DialogHeader>

          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <Label htmlFor="name">নাম</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                placeholder="মোঃ রফিকুল ইসলাম"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="bloodGroup">রক্তের গ্রুপ</Label>
                <Select
                  value={formData.bloodGroup}
                  onValueChange={(value) => setFormData({ ...formData, bloodGroup: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="নির্বাচন করুন" />
                  </SelectTrigger>
                  <SelectContent>
                    {bloodGroups.map((group) => (
                      <SelectItem key={group} value={group}>
                        {group}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="grid gap-2">
                <Label htmlFor="phone">ফোন নম্বর</Label>
                <Input
                  id="phone"
                  value={formData.phone}
                  onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                  placeholder="০১৭১২৩৪৫৬৭৮"
                />
              </div>
            </div>

            <div className="grid gap-2">
              <Label htmlFor="address">ঠিকানা</Label>
              <Input
                id="address"
                value={formData.address}
                onChange={(e) => setFormData({ ...formData, address: e.target.value })}
                placeholder="শীতলকার, বরিশাল"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="hospital">হাসপাতাল (ঐচ্ছিক)</Label>
              <Input
                id="hospital"
                value={formData.hospital}
                onChange={(e) => setFormData({ ...formData, hospital: e.target.value })}
                placeholder="শীতলকার সরকারি হাসপাতাল"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="status">স্ট্যাটাস</Label>
              <Select
                value={formData.status}
                onValueChange={(value: BloodDonor["status"]) =>
                  setFormData({ ...formData, status: value })
                }
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="published">প্রকাশিত</SelectItem>
                  <SelectItem value="pending">অপেক্ষমান</SelectItem>
                  <SelectItem value="draft">খসড়া</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
              বাতিল
            </Button>
            <Button onClick={handleSubmit} className="btn-gradient-secondary">
              {editingDonor ? "আপডেট করুন" : "যুক্ত করুন"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AdminLayout>
  );
}
