import { AdminLayout } from "@/components/admin/AdminLayout";
import { DataTable, Column } from "@/components/admin/DataTable";
import { useState } from "react";
import { toast } from "sonner";
import { Phone, MapPin, Ambulance as AmbulanceIcon, Car } from "lucide-react";

interface AmbulanceData {
  id: string;
  name: string;
  driverName: string;
  phone: string;
  area: string;
  vehicleNumber: string;
  status: "published" | "pending" | "draft";
}

const mockAmbulances: AmbulanceData[] = [
  {
    id: "1",
    name: "শীতলকার অ্যাম্বুলেন্স সার্ভিস",
    driverName: "মোঃ করিম",
    phone: "০১৭১২৩৪৫৬৭৮",
    area: "শীতলকার, বরিশাল",
    vehicleNumber: "বরিশাল মেট্রো ক-১২৩৪",
    status: "published",
  },
  {
    id: "2",
    name: "ফাস্ট এইড অ্যাম্বুলেন্স",
    driverName: "রহিম উদ্দিন",
    phone: "০১৮১২৩৪৫৬৭৮",
    area: "বানারীপাড়া",
    vehicleNumber: "বরিশাল মেট্রো খ-৫৬৭৮",
    status: "published",
  },
  {
    id: "3",
    name: "লাইফ লাইন অ্যাম্বুলেন্স",
    driverName: "জাহিদ হাসান",
    phone: "০১৯১২৩৪৫৬৭৮",
    area: "উজিরপুর",
    vehicleNumber: "বরিশাল মেট্রো গ-৯০১২",
    status: "pending",
  },
];

const columns: Column<AmbulanceData>[] = [
  {
    key: "name",
    header: "সার্ভিস",
    render: (item) => (
      <div className="flex items-center gap-3">
        <div className="h-10 w-10 rounded-full bg-warning/10 flex items-center justify-center">
          <AmbulanceIcon className="h-5 w-5 text-warning" />
        </div>
        <div>
          <p className="font-medium">{item.name}</p>
          <p className="text-sm text-muted-foreground">{item.driverName}</p>
        </div>
      </div>
    ),
  },
  {
    key: "vehicleNumber",
    header: "গাড়ি নম্বর",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Car className="h-4 w-4 text-muted-foreground" />
        <span className="font-mono">{item.vehicleNumber}</span>
      </div>
    ),
  },
  {
    key: "area",
    header: "এলাকা",
    render: (item) => (
      <div className="flex items-center gap-2">
        <MapPin className="h-4 w-4 text-muted-foreground" />
        <span>{item.area}</span>
      </div>
    ),
  },
  {
    key: "phone",
    header: "ফোন",
    render: (item) => (
      <div className="flex items-center gap-2">
        <Phone className="h-4 w-4 text-muted-foreground" />
        <span>{item.phone}</span>
      </div>
    ),
  },
  { key: "status", header: "স্ট্যাটাস" },
];

export default function AmbulancePage() {
  const [ambulances, setAmbulances] = useState<AmbulanceData[]>(mockAmbulances);

  const handleDelete = (item: AmbulanceData) => {
    setAmbulances(ambulances.filter((a) => a.id !== item.id));
    toast.success("অ্যাম্বুলেন্স তথ্য সফলভাবে মুছে ফেলা হয়েছে");
  };

  return (
    <AdminLayout
      title="অ্যাম্বুলেন্স ম্যানেজমেন্ট"
      description="সকল অ্যাম্বুলেন্স সার্ভিসের তালিকা দেখুন এবং পরিচালনা করুন"
    >
      <DataTable
        data={ambulances}
        columns={columns}
        onAdd={() => toast.info("নতুন অ্যাম্বুলেন্স যুক্ত করার ফর্ম আসবে")}
        onEdit={(item) => toast.info(`${item.name} সম্পাদনা করা হচ্ছে`)}
        onDelete={handleDelete}
        addButtonText="নতুন অ্যাম্বুলেন্স যুক্ত করুন"
        searchPlaceholder="অ্যাম্বুলেন্স বা এলাকা দিয়ে খুঁজুন..."
      />
    </AdminLayout>
  );
}
