import { ReactNode } from "react";
import { cn } from "@/lib/utils";
import { LucideIcon, TrendingUp, TrendingDown } from "lucide-react";

interface StatCardProps {
  title: string;
  value: string | number;
  icon: LucideIcon;
  trend?: {
    value: number;
    isPositive: boolean;
  };
  variant?: "default" | "primary" | "secondary" | "success" | "warning";
  className?: string;
}

const variantStyles = {
  default: "bg-card border-border",
  primary: "bg-primary/5 border-primary/20 text-primary",
  secondary: "bg-secondary/5 border-secondary/20 text-secondary",
  success: "bg-success/5 border-success/20 text-success",
  warning: "bg-warning/5 border-warning/20 text-warning",
};

export function StatCard({
  title,
  value,
  icon: Icon,
  trend,
  variant = "default",
  className,
}: StatCardProps) {
  return (
    <div
      className={cn(
        "stat-card border rounded-xl p-6 transition-all duration-300 hover:shadow-lg",
        variantStyles[variant],
        className
      )}
    >
      <div className="flex items-start justify-between">
        <div>
          <p className="text-sm font-medium text-muted-foreground">{title}</p>
          <p className="text-3xl font-bold mt-2 text-foreground">{value}</p>
          
          {trend && (
            <div
              className={cn(
                "flex items-center gap-1 mt-2 text-sm font-medium",
                trend.isPositive ? "text-success" : "text-destructive"
              )}
            >
              {trend.isPositive ? (
                <TrendingUp className="h-4 w-4" />
              ) : (
                <TrendingDown className="h-4 w-4" />
              )}
              <span>{Math.abs(trend.value)}%</span>
              <span className="text-muted-foreground font-normal">গত মাসের চেয়ে</span>
            </div>
          )}
        </div>
        
        <div
          className={cn(
            "h-12 w-12 rounded-xl flex items-center justify-center",
            variant === "default" ? "bg-muted" : "bg-current/10"
          )}
        >
          <Icon className={cn("h-6 w-6", variant === "default" ? "text-muted-foreground" : "")} />
        </div>
      </div>
    </div>
  );
}
