import { useState } from "react";
import { NavLink, useLocation } from "react-router-dom";
import {
  LayoutDashboard,
  Stethoscope,
  Droplets,
  Ambulance,
  Flame,
  Shield,
  Scale,
  Building2,
  Users,
  Gamepad2,
  UserCheck,
  Image,
  Heart,
  Settings,
  ChevronDown,
  Menu,
  X,
  LogOut,
  Bell,
} from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";

interface NavItem {
  title: string;
  href: string;
  icon: React.ComponentType<{ className?: string }>;
  badge?: number;
}

interface NavGroup {
  title: string;
  items: NavItem[];
}

const navGroups: NavGroup[] = [
  {
    title: "মূল মেনু",
    items: [
      { title: "ড্যাশবোর্ড", href: "/", icon: LayoutDashboard },
    ],
  },
  {
    title: "সার্ভিস ম্যানেজমেন্ট",
    items: [
      { title: "ডক্টর", href: "/doctors", icon: Stethoscope },
      { title: "রক্তদাতা", href: "/blood-donors", icon: Droplets, badge: 5 },
      { title: "অ্যাম্বুলেন্স", href: "/ambulance", icon: Ambulance },
      { title: "ফায়ার সার্ভিস", href: "/fire-service", icon: Flame },
      { title: "পুলিশ", href: "/police", icon: Shield },
      { title: "আইনজীবী", href: "/lawyers", icon: Scale },
      { title: "ইমাম সাহেব", href: "/imams", icon: Building2 },
    ],
  },
  {
    title: "সদস্য ও খেলোয়াড়",
    items: [
      { title: "চেয়ারম্যান/মেম্বার", href: "/members", icon: Users },
      { title: "খেলোয়াড়", href: "/players", icon: Gamepad2 },
      { title: "ক্লাব সদস্য", href: "/club-users", icon: UserCheck, badge: 3 },
    ],
  },
  {
    title: "কন্টেন্ট",
    items: [
      { title: "স্লাইডার", href: "/slider", icon: Image },
      { title: "ডোনেশন", href: "/donations", icon: Heart },
    ],
  },
  {
    title: "সিস্টেম",
    items: [
      { title: "সেটিংস", href: "/settings", icon: Settings },
    ],
  },
];

export function AdminSidebar() {
  const [isCollapsed, setIsCollapsed] = useState(false);
  const [openGroups, setOpenGroups] = useState<string[]>(navGroups.map(g => g.title));
  const location = useLocation();

  const toggleGroup = (title: string) => {
    setOpenGroups(prev =>
      prev.includes(title)
        ? prev.filter(t => t !== title)
        : [...prev, title]
    );
  };

  return (
    <>
      {/* Mobile Overlay */}
      {!isCollapsed && (
        <div
          className="fixed inset-0 bg-foreground/20 backdrop-blur-sm z-40 lg:hidden"
          onClick={() => setIsCollapsed(true)}
        />
      )}

      {/* Sidebar */}
      <aside
        className={cn(
          "fixed left-0 top-0 z-50 h-screen sidebar-gradient text-sidebar-foreground transition-all duration-300 flex flex-col",
          isCollapsed ? "w-0 lg:w-16" : "w-64"
        )}
      >
        {/* Header */}
        <div className="flex h-16 items-center justify-between px-4 border-b border-sidebar-border">
          {!isCollapsed && (
            <div className="flex items-center gap-3">
              <div className="h-9 w-9 rounded-lg bg-primary flex items-center justify-center">
                <span className="text-primary-foreground font-bold text-lg">শী</span>
              </div>
              <div>
                <h1 className="font-bold text-sm">শীতলকার</h1>
                <p className="text-xs text-sidebar-foreground/60">অ্যাডমিন প্যানেল</p>
              </div>
            </div>
          )}
          <Button
            variant="ghost"
            size="icon"
            className="text-sidebar-foreground hover:bg-sidebar-accent"
            onClick={() => setIsCollapsed(!isCollapsed)}
          >
            {isCollapsed ? <Menu className="h-5 w-5" /> : <X className="h-5 w-5" />}
          </Button>
        </div>

        {/* Navigation */}
        <nav className={cn("flex-1 overflow-y-auto scrollbar-thin py-4", isCollapsed && "hidden lg:block")}>
          {navGroups.map((group) => (
            <Collapsible
              key={group.title}
              open={openGroups.includes(group.title)}
              onOpenChange={() => toggleGroup(group.title)}
              className="mb-2"
            >
              {!isCollapsed && (
                <CollapsibleTrigger className="flex w-full items-center justify-between px-4 py-2 text-xs font-medium uppercase tracking-wider text-sidebar-foreground/50 hover:text-sidebar-foreground/80">
                  {group.title}
                  <ChevronDown
                    className={cn(
                      "h-4 w-4 transition-transform",
                      openGroups.includes(group.title) && "rotate-180"
                    )}
                  />
                </CollapsibleTrigger>
              )}

              <CollapsibleContent>
                <ul className="space-y-1 px-2">
                  {group.items.map((item) => {
                    const isActive = location.pathname === item.href;
                    return (
                      <li key={item.href}>
                        <NavLink
                          to={item.href}
                          className={cn(
                            "flex items-center gap-3 rounded-lg px-3 py-2.5 text-sm font-medium transition-all",
                            isActive
                              ? "bg-sidebar-primary text-sidebar-primary-foreground shadow-md"
                              : "text-sidebar-foreground/80 hover:bg-sidebar-accent hover:text-sidebar-foreground"
                          )}
                        >
                          <item.icon className={cn("h-5 w-5 flex-shrink-0", isCollapsed && "mx-auto")} />
                          {!isCollapsed && (
                            <>
                              <span className="flex-1">{item.title}</span>
                              {item.badge && (
                                <span className="flex h-5 min-w-[20px] items-center justify-center rounded-full bg-secondary text-[10px] font-bold text-secondary-foreground">
                                  {item.badge}
                                </span>
                              )}
                            </>
                          )}
                        </NavLink>
                      </li>
                    );
                  })}
                </ul>
              </CollapsibleContent>
            </Collapsible>
          ))}
        </nav>

        {/* Footer */}
        {!isCollapsed && (
          <div className="border-t border-sidebar-border p-4">
            <div className="flex items-center gap-3 mb-3">
              <div className="h-10 w-10 rounded-full bg-sidebar-accent flex items-center justify-center">
                <span className="text-sm font-medium">এড</span>
              </div>
              <div className="flex-1 min-w-0">
                <p className="text-sm font-medium truncate">অ্যাডমিন</p>
                <p className="text-xs text-sidebar-foreground/60 truncate">admin@shitolkar.top</p>
              </div>
              <Button
                variant="ghost"
                size="icon"
                className="text-sidebar-foreground/60 hover:text-sidebar-foreground hover:bg-sidebar-accent"
              >
                <Bell className="h-4 w-4" />
              </Button>
            </div>
            <Button
              variant="ghost"
              className="w-full justify-start gap-2 text-sidebar-foreground/60 hover:text-secondary hover:bg-sidebar-accent"
            >
              <LogOut className="h-4 w-4" />
              লগআউট
            </Button>
          </div>
        )}
      </aside>

      {/* Mobile Toggle Button */}
      <Button
        variant="outline"
        size="icon"
        className="fixed left-4 top-4 z-30 lg:hidden"
        onClick={() => setIsCollapsed(false)}
      >
        <Menu className="h-5 w-5" />
      </Button>
    </>
  );
}
