<?php
// DB + global app config
require_once __DIR__ . '/jsondb.php';
require_once __DIR__ . '/../../config.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

function ensure_admin_initialized(): void {
    // Values come from config.php
    global $DEFAULT_ADMIN_EMAIL, $DEFAULT_ADMIN_PASSWORD;
    $admin = db_read('admin_user.json', null);
    if (!$admin || !is_array($admin) || empty($admin['email'])) {
        $admin = [
            'email' => $DEFAULT_ADMIN_EMAIL,
            'password_hash' => password_hash($DEFAULT_ADMIN_PASSWORD, PASSWORD_DEFAULT),
            'created_at' => date('c'),
            'updated_at' => date('c')
        ];
        db_write('admin_user.json', $admin);
    }
}

function is_logged_in(): bool {
    return !empty($_SESSION['admin_logged_in']);
}

// Backward-compatible helper (some templates call this name)
function is_admin_logged_in(): bool {
    return is_logged_in();
}


function require_login(): void {
    if (!is_logged_in()) {
        header('Location: login.php');
        exit;
    }
}

function csrf_token(): string {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
    }
    return $_SESSION['csrf_token'];
}

function csrf_check(): void {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $token = $_POST['csrf_token'] ?? '';
        if (!$token || $token !== ($_SESSION['csrf_token'] ?? '')) {
            http_response_code(403);
            die('Invalid CSRF token');
        }
    }
}

/**
 * Attempt admin login.
 * Returns true on success, false on failure.
 */
function login_admin(string $email, string $password): bool {
    global $DEFAULT_ADMIN_EMAIL, $DEFAULT_ADMIN_PASSWORD;
    $email = strtolower(trim($email));
    $password = (string)$password;

    if ($email === '' || $password === '') return false;

    // Make sure default admin exists
    ensure_admin_initialized();

    $admin = db_read('admin_user.json', [
        'email' => $DEFAULT_ADMIN_EMAIL,
        'password_hash' => password_hash($DEFAULT_ADMIN_PASSWORD, PASSWORD_DEFAULT),
        'created_at' => date('c'),
    ]);

    $savedEmail = strtolower(trim($admin['email'] ?? ''));
    $savedHash  = $admin['password_hash'] ?? '';

    if ($savedEmail !== $email) return false;
    if (!is_string($savedHash) || $savedHash === '') return false;
    if (!password_verify($password, $savedHash)) return false;

    // Session hardening
    if (session_status() !== PHP_SESSION_ACTIVE) {
        @session_start();
    }
    @session_regenerate_id(true);

    $_SESSION['admin_logged_in'] = true;
    $_SESSION['admin_email']     = $savedEmail;
    $_SESSION['admin_login_at']  = time();

    return true;
}

/**
 * Logout admin safely.
 */
function logout_admin(): void {
    if (session_status() !== PHP_SESSION_ACTIVE) {
        @session_start();
    }

    $_SESSION = [];

    // Remove session cookie if exists
    if (ini_get('session.use_cookies')) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params['path'] ?? '/',
            $params['domain'] ?? '',
            (bool)($params['secure'] ?? false),
            (bool)($params['httponly'] ?? true)
        );
    }

    @session_destroy();
}

/**
 * Returns currently logged-in admin info for the header / UI.
 * Format: ['email' => 'admin@localhost']
 */
function current_admin(): array {
    if (session_status() !== PHP_SESSION_ACTIVE) {
        @session_start();
    }

    // If not logged in, return safe defaults
    if (!is_admin_logged_in()) {
        // try fallback from stored admin file
        $saved = db_read(ADMIN_FILE, [
            'email' => 'admin@localhost',
            'password_hash' => password_hash('admin123', PASSWORD_DEFAULT)
        ]);
        return [
            'email' => $saved['email'] ?? 'admin@localhost'
        ];
    }

    // Primary session source
    $email = $_SESSION['admin_email'] ?? '';

    // Secondary fallback: admin file (in case session missing)
    if ($email === '') {
        $saved = db_read(ADMIN_FILE, null);
        $email = is_array($saved) ? ($saved['email'] ?? '') : '';
    }

    if ($email === '') {
        $email = 'admin@localhost';
    }

    return [
        'email' => $email
    ];
}

ensure_admin_initialized();